package de.xam.tupleinf.impl;

import java.util.Set;

import org.xydra.base.Base;
import org.xydra.base.XId;
import org.xydra.index.IEntrySet;
import org.xydra.index.IMapIndex;
import org.xydra.index.IMapSetIndex;
import org.xydra.index.impl.FastEntrySetFactory;
import org.xydra.index.impl.MapIndex;
import org.xydra.index.impl.MapSetIndex;
import org.xydra.index.iterator.Iterators;
import org.xydra.log.api.Logger;
import org.xydra.log.api.LoggerFactory;

import de.xam.tupleinf.IInverseTransitiveTupleIndex;

/**
 * 'STTI', a friend of {@link IInverseTransitiveTupleIndex} (ITTI)
 *
 * Handles relations which are symmetric & transitive. .
 *
 * <h3>sameAs</h3> 'sameAs' is transitive and symmetric (inverse to itself)
 *
 * For these triples
 *
 * <pre>
 * --- base:
 * a p b
 * p sameAs q
 * q sameAs r
 * b sameAs c
 * --- additional inf:
 * a q b
 * a q c
 * a r b
 * a r c
 * c sameAs b (inverse)
 * p sameAs r (transitive)
 * q sameAs p (inverse)
 * q sameAs r (inverse+transitive)
 * r sameAs p (inverse+transitive)
 * r sameAs q (inverse)
 * </pre>
 *
 * The ITTI for 'sameAs' looks like this:
 *
 * <pre>
 * base:
 * (p -> q)
 * (q -> r)
 * (b -> c)
 * inf:
 * (b -> c)
 * (c -> b)
 * (p -> q, r)
 * (q -> r, p)
 * (r -> p, q)
 * </pre>
 *
 * hence it represents two sameAs-families (b,c) and (p,q,r). For each family,
 * we need two indexes: from family to canonical and from canonical to all
 * family members.
 *
 * @author xamde
 *
 */
public class SymmetricTransitiveTupleIndex {

	private static final Logger log = LoggerFactory.getLogger(SymmetricTransitiveTupleIndex.class);

	public static void main(final String[] args) {
		System.out.println(selectCanonical(Base.toId("aaa"), Base.toId("bbb")));
	}

	private static XId selectCanonical(final XId a, final XId b) {
		assert !a.equals(b);
		if (a.compareTo(b) < 0) {
			return a;
		} else {
			return b;
		}
	}

	/** family does not include the canonical */
	private final IMapSetIndex<XId, XId> canonical2family = new MapSetIndex<XId, XId>(
			new FastEntrySetFactory<XId>());

	/** canonical maps to itself */
	private final IMapIndex<XId, XId> family2canonical = new MapIndex<XId, XId>();

	/** the famous dirty flag */
	protected boolean isDirtyTransitives = false;

	/**
	 * This method might break apart a family into two; shrink an existing
	 * family or remove a family alltogether.
	 *
	 * @param a
	 * @param b
	 */
	public void deIndex_s_sameAs_o(final XId a, final XId b) {
		assert a != null;
		assert b != null;
		if (a.equals(b)) {
			return;
		}

		/**
		 * cases:
		 *
		 * <pre>
		 * [1] canonicalOfA= ----, canonicalOfB= ----
		 *    -> WARN, do nothing
		 * [2] canonicalOfA= ----, canonicalOfB= yes
		 *    -> WARN, do nothing
		 * [3] canonicalOfA= yes,  canonicalOfB= ----
		 *    -> WARN, do nothing
		 * [4] canonicalOfA= yes,  canonicalOfB= yes
		 *    [4.1] same canonical
		 *       [x]
		 *          -> @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
		 *    [4.2] different canonical
		 *          -> WARN, do nothing
		 * </pre>
		 *
		 **/
		final XId canonicalOfA = this.family2canonical.lookup(a);
		final XId canonicalOfB = this.family2canonical.lookup(b);
		if (canonicalOfA == null || canonicalOfB == null) {
			log.warn("canonical(" + a + ")=" + canonicalOfA + " canonical(" + b + ")="
					+ canonicalOfB);
			return;
		}

		if (canonicalOfA != canonicalOfB) {
			log.trace("different canonicals:" + "canonical(" + a + ")=" + canonicalOfA
					+ " canonical(" + b + ")=" + canonicalOfB);
			return;
		}

		final XId canonical = canonicalOfA;
		assert canonical == canonicalOfB;

	}

	@SuppressWarnings("unused")
	private void deIndexCanonical(final XId canonical, final XId familyMember) {
		this.canonical2family.deIndex(canonical, familyMember);
		this.family2canonical.deIndex(familyMember);
	}

	/**
	 * @param id
	 * @return @CanBeNull if id is not aliased in any way
	 */
	public XId getCanononical(final XId id) {
		return this.family2canonical.lookup(id);
	}

	/**
	 * This triple might create a new family, extend an existing family or merge
	 * two families together. Or just be redundant anyway.
	 *
	 * It might cause a new canonical to be chosen or keep the existing
	 * canonical.
	 *
	 * @param a
	 * @param b
	 */
	public void index_s_sameAs_o(final XId a, final XId b) {
		assert a != null;
		assert b != null;
		if (a.equals(b)) {
			return;
		}

		/**
		 * cases:
		 *
		 * <pre>
		 * [1] canonicalOfA= ----, canonicalOfB= ----
		 *    -> select canonical, add other(a/b)
		 * [2] canonicalOfA= ----, canonicalOfB= yes
		 *    -> canonicalOfB.add(a)
		 * [3] canonicalOfA= yes,  canonicalOfB= ----
		 *    -> canonicalOfA.add(b)
		 * [4] canonicalOfA= yes,  canonicalOfB= yes
		 *   [4.1] same canonical
		 *      -> nothing to do
		 *   [4.2] different canonical
		 *      -> select canonical, move all losers to the winner
		 * </pre>
		 *
		 **/
		final XId canonicalOfA = this.family2canonical.lookup(a);
		final XId canonicalOfB = this.family2canonical.lookup(b);
		if (canonicalOfA == null) {
			if (canonicalOfB == null) {
				/* case 1 */
				final XId winner = selectCanonical(a, b);
				if (winner == a) {
					indexCanonical(a, b);
				} else {
					indexCanonical(b, a);
				}
			} else {
				/* case 2 */
				indexCanonical(canonicalOfB, a);
			}
		} else {
			if (canonicalOfB == null) {
				/* case 3 */
				indexCanonical(canonicalOfA, b);
			} else {
				/* case 4 */
				if (canonicalOfA.equals(canonicalOfB)) {
					/* case 4.1 */
					// do nothing
				} else {
					/* case 4.2 */
					final XId winner = selectCanonical(canonicalOfA, canonicalOfB);
					if (winner == canonicalOfA) {
						migrate(canonicalOfB, canonicalOfA);
					} else {
						assert winner == canonicalOfB;
						migrate(canonicalOfB, canonicalOfA);
					}
				}
			}
		}
	}

	private void indexCanonical(final XId canonical, final XId familyMember) {
		this.canonical2family.index(canonical, familyMember);
		this.family2canonical.index(familyMember, canonical);
	}

	/**
	 * Migrate the complete family of canonicalA to be family members of
	 * canonicalB.
	 *
	 * @param canonicalA
	 * @param canonicalB
	 */
	private void migrate(final XId canonicalA, final XId canonicalB) {
		final IEntrySet<XId> familyB = this.canonical2family.lookup(canonicalB);
		// copy set to avoid concurrency issues
		final Set<XId> membersB = Iterators.toSet(familyB.iterator());
		for (final XId b : membersB) {
			this.family2canonical.deIndex(b);
			this.family2canonical.index(canonicalA, b);
		}
		this.canonical2family.deIndex(canonicalB);
	}

}
