package de.xam.tupleinf.impl;

import java.util.Iterator;

import org.xydra.index.iterator.ITransformer;
import org.xydra.index.iterator.Iterators;
import org.xydra.index.query.Constraint;
import org.xydra.index.query.KeyEntryTuple;
import org.xydra.index.query.Wildcard;
import org.xydra.log.api.Logger;
import org.xydra.log.api.LoggerFactory;
import org.xydra.sharedutils.DebugUtils;

import de.xam.tupleinf.IInverseTransitiveTupleIndex;
import de.xam.tupleinf.InfLayerRead;

/**
 * Bases class handles logic of flags (knowledge explicitly asserted) and effective... settings (knowledge deduced from
 * facts). E.g. if an index is transitive and its inverse is not, then effectively, the inverse index is transitive,
 * too. However, when split apart, the first one remains transitive, the second one becomes normal again.
 *
 * Basic implementation pattern: A method called <code>foo</code> will first determine who the primary is, and then call
 * <code>foo_primary</code> on the primary. Now the ITTI knows it is primary and can proceed to do things. This is often
 * done with methods like <code>do_foo</code> which just executes the action. See the code of {@link #clearInf()} as an
 * example.
 *
 * 'foo'
 *
 * 'do__foo' methods may never read/write/check the inverse index at all.
 *
 * @author xamde
 *
 * @param <T>
 */
public abstract class ITTI_Base<T> implements IInverseTransitiveTupleIndex<T> {

	private static final Logger log = LoggerFactory.getLogger(ITTI_Base.class);

	@SuppressWarnings("rawtypes")
	protected static final ITransformer TUPLE_INVERTER = new ITransformer<KeyEntryTuple, KeyEntryTuple>() {

		@SuppressWarnings("unchecked")
		@Override
		public KeyEntryTuple transform(final KeyEntryTuple t) {
			return new KeyEntryTuple(t.getSecond(), t.getFirst());
		}
	};

	@SuppressWarnings("unchecked")
	protected static <T> Iterator<KeyEntryTuple<T, T>> invert(final Iterator<KeyEntryTuple<T, T>> it) {
		return Iterators.transform(it, TUPLE_INVERTER);
	}

	protected final Constraint<T> ANY = new Wildcard<T>();

	private final boolean concurrent;

	/**
	 * The current effective value if this index is symmetric or not. This value is only accurate for the primary index.
	 * Undefined state for a secondary index.
	 */
	protected boolean effectiveSymmetric;

	/**
	 * The current effective value if this index is transitive or not. This value is only accurate for the primary
	 * index. Undefined state for a secondary index.
	 */
	protected boolean effectiveTransitive;

	/**
	 * the flag set by the user about this index. It is remembered to restored a sane state after the inverse index is
	 * removed again (if one was ever attached).
	 */
	protected boolean flagSymmetric;

	/**
	 * the flag set by the user about this index. It is remembered to restored a sane state after the inverse index is
	 * removed again (if one was ever attached).
	 */
	protected boolean flagTransitive;

	/**
	 * @CanBeNull for symmetric or stand-alone properties
	 */
	private ITTI_Base<T> inverseIndex;

	/** the famous dirty flag */
	protected boolean isDirtyInf = false;

	/**
	 * true if this index is the primary index or the inverse index is null
	 */
	private boolean isPrimary;

	/** this is the p of spo */
	protected final T p;

	/**
	 * @param p
	 * @param concurrent iff true, run in a concurrency-safe way (can either be a fail-fast or a true concurrency mode)
	 */
	public ITTI_Base(final T p, final boolean concurrent) {
		this.p = p;
		this.concurrent = concurrent;
		this.effectiveSymmetric = false;
		this.effectiveTransitive = false;
		this.flagSymmetric = false;
		this.flagTransitive = false;
		this.inverseIndex = null;
		this.isDirtyInf = false;
		this.isPrimary = true;
	}

	protected final void checkAssertions() {
		/** #isSymmetric implies #isPrimary & !#hasInverse */
		if (this.effectiveSymmetric) {
			assert this.isPrimary;
			assert this.inverseIndex == null;
		}
		if (this.inverseIndex != null) {
			assert!this.effectiveSymmetric;
		}
		if (this.effectiveTransitive && this.inverseIndex != null) {
			this.inverseIndex.isTransitive();
		}
		if (this.isPrimary && this.inverseIndex != null) {
			assert!this.inverseIndex.isPrimary;
		}
		if (!this.isPrimary) {
			assert this.inverseIndex != null : "a secondary index always has an inverse";
		}
	}

	@Override
	public final void clearInf() {
		if (isSecondary()) {
			getInverseIndex().primary__clearInf();
		} else {
			primary__clearInf();
		}
	}

	@Override
	public final boolean contains(final Constraint<T> cA, final Constraint<T> cB, final InfLayerRead infLayer) {
		if (log.isDebugEnabled()) {
			log.debug("Tuplequery " + this.p + ".(" + cA + "," + cB + ")");
		}

		if (isSecondary()) {
			return getInverseIndex().primary__contains(cB, cA,infLayer);
		} else {
			return primary__contains(cA, cB,infLayer);
		}
	}

	@Override
	public final boolean contains(final T a, final T b, final InfLayerRead infLayer) {
		if (log.isDebugEnabled()) {
			log.debug("Tuplequery " + this.p + ".(" + a + "," + b + ")");
		}

		if (isSecondary()) {
			return getInverseIndex().primary__contains(b, a,infLayer);
		} else {
			return primary__contains(a, b,infLayer);
		}
	}

	@Override
	public final boolean deIndex(final T a, final T b) {
		if (isSecondary()) {
			return getInverseIndex().primary__deIndex(b, a);
		} else {
			return primary__deIndex(a, b);
		}
	}

	/** Delete all indexed tuples. Does not imply clearInf. */
	protected abstract void do__clearBase();

	/** Delete all inferred tuples */
	protected abstract void do__clearInfd();

	/**
	 * Dumps index content to given {@link StringBuilder} for debugging
	 *
	 * @param buf
	 */
	protected abstract void do__contentToString(StringBuilder buf);

	protected final void do__dump() {
		log.info(do__metaDataToString());
		do__dumpContent();
	}

	/**
	 * Dumps content to log.info()
	 */
	protected abstract void do__dumpContent();

	/**
	 * This is set exclusively by implementations
	 */
	protected final void do__markInferenceAsComplete() {
		this.isDirtyInf = false;
	}

	protected final void do__markInferenceAsDirty() {
		this.isDirtyInf = true;
	}

	protected final String do__metaDataToString() {
		return "p='" + getP() + "' -> " +

		"inv=" + (hasInverse() ? "'" + getInverseIndex().getP() + "'" : "--") + " " +

		DebugUtils.flagToString("primary", this.isPrimary) + " " +

		DebugUtils.flagToString("trans", this.effectiveTransitive) + "" +

		"(" + DebugUtils.flagToString("flag", this.flagTransitive) + ") " +

		DebugUtils.flagToString("symm", this.effectiveSymmetric) + "" +

		"(" + DebugUtils.flagToString("flag", this.flagSymmetric) + ") " +

		DebugUtils.flagToString("dirty", this.isDirtyInf);
	}

	/**
	 * Removes an inverse ITTI and deletes all indexed tuples. Tuples need to be re-indexed, since it is unclear which
	 * tuple belonged where.
	 *
	 * E.g. when we the ITTI contains (s,p,o) we don't know if it was stated as (s,p,o) or (o,q,s) (assuming p and q had
	 * been inverse).
	 */
	private void do__setNoInverseIndexAndClearAll() {
		this.inverseIndex = null;
		this.isPrimary = true;
		// restore
		this.effectiveSymmetric = this.flagSymmetric;
		this.effectiveTransitive = this.flagTransitive;

		// this clears everything!
		do__clearBase();
		do__clearInfd();
	}

	/**
	 * @param symmetric
	 * @return true iff old != new value
	 */
	private final boolean do__setSymmetric(final boolean symmetric) {
		final boolean changes = this.flagSymmetric != symmetric;
		// always remember the flag
		this.flagSymmetric = symmetric;
		return changes;
	}

	// inf = dirty, primary = false
	@Override
	public final void do__setToSecondaryWithPrimaryInverseIndex(
			final IInverseTransitiveTupleIndex<T> primaryInverseIndex) {
		assert primaryInverseIndex != null;
		if (log.isDebugEnabled()) {
			log.debug("Set '" + getP() + "'.inverseIndex = '" + primaryInverseIndex.getP() + "'");
		}

		this.inverseIndex = (ITTI_Base<T>) primaryInverseIndex;
		do__markInferenceAsDirty();
		this.isPrimary = false;
		// don't link anything any more
	}

	/**
	 * @param transitive
	 * @return true if new value != old value
	 */
	private final boolean do__setTransitive(final boolean transitive) {
		final boolean changes = this.flagTransitive != transitive;
		// always remember the flag
		this.flagTransitive = transitive;
		return changes;
	}

	protected final String do__toString() {
		final StringBuilder buf = new StringBuilder();
		buf.append(do__metaDataToString() + "\n");
		do__contentToString(buf);
		return buf.toString();
	}

	protected abstract Iterator<KeyEntryTuple<T, T>> do__tupleIterator(Constraint<T> cA, Constraint<T> cB);

	@Override
	public final void dump(final boolean dumpAll) {
		if (dumpAll && isSecondary()) {
			if (log.isInfoEnabled()) {
				log.info("...see at inverse '" + getInverseIndex().getP() + "'");
			}
			return;
		}

		if (isSecondary()) {
			getInverseIndex().primary__dump();
		} else {
			primary__dump();
		}
	}

	@Override
	public final ITTI_Base<T> getInverseIndex() {
		return this.inverseIndex;
	}

	@Override
	public final T getP() {
		return this.p;
	}

	@Override
	public final boolean hasInverse() {
		if (getInverseIndex() != null) {
			assert!this.effectiveSymmetric;
			return true;
		} else {
			return false;
		}
	}

	/**
	 * @return true if this ITTI has a secondary index, i.e. there is one and this one is primary.
	 */
	protected final boolean hasSecondary() {
		return hasInverse() && isPrimary();
	}

	@Override
	public final boolean index(final T a, final T b) {
		if (isSecondary()) {
			return getInverseIndex().primary__index(b, a);
		} else {
			return primary__index(a, b);
		}
	}

	@Override
	public void inferAll() {
		checkAssertions();
		primary__inference();
	}

	/**
	 * @return true if this index was created in a concurrency-safe way (can either be a fail-fast or a true concurrency
	 *         mode)
	 */
	public final boolean isConcurrent() {
		return this.concurrent;
	}

	@Override
	public final boolean isEmpty() {
		if (isSecondary()) {
			return getInverseIndex().primary__isEmpty();
		} else {
			return primary__isEmpty();
		}
	}

	protected final boolean isInferenceComplete() {
		return !this.isDirtyInf;
	}

	@Override
	public boolean isPrimary() {
		return this.isPrimary;
	}

	@Override
	public final boolean isSecondary() {
		assert isPrimary() || this.inverseIndex != null;
		return !isPrimary();
	}

	@Override
	public final boolean isSymmetric() {
		checkAssertions();
		return isPrimary() ? primary__isSymmetric() : getInverseIndex().primary__isSymmetric();
	}

	@Override
	public final boolean isTransitive() {
		return isPrimary() ? primary__isTransitive() : getInverseIndex().primary__isTransitive();
	}

	/**
	 * Used for re-indexing content when an index looses its inverse index
	 *
	 * @return all tuples in base index
	 */
	public abstract Iterator<KeyEntryTuple<T, T>> primary__baseTupleIterator();

	/** Clear base and inf */
	protected abstract void primary__clearAll();

	protected final void primary__clearInf() {
		do__clearInfd();
		if (hasSecondary()) {
			getInverseIndex().do__clearInfd();
		}
	}

	/**
	 * @param cA
	 * @NeverNull
	 * @param cB
	 * @NeverNull
	 * @return true if primary index contains tuples matching (cA,cB)
	 */
	protected abstract boolean primary__contains(Constraint<T> cA, Constraint<T> cB, InfLayerRead infLayer);

	/**
	 * @param a
	 * @NeverNull
	 * @param b
	 * @NeverNull
	 * @return true if primary index contains tuples matching (cB,cA)
	 */
	protected abstract boolean primary__contains(T a, T b, InfLayerRead infLayer);

	/**
	 * @param b
	 * @param a
	 * @return true if there was in fact such a tuple
	 */
	protected abstract boolean primary__deIndex(T b, T a);

	protected final void primary__dump() {
		if (!isInferenceComplete()) {
			primary__inference();
		}

		do__dump();
		if (hasSecondary()) {
			log.info("=== Secondary: ");
			getInverseIndex().do__dump();
		}
	}

	/** Runs inference algorithm if dirty flag is set */
	protected final void primary__ensureInference() {
		if (primary__usesInference() && this.isDirtyInf) {
			if (log.isDebugEnabled()) {
				log.debug("dirty flag set -> inferring");
			}
			primary__inference();
		}
	}

	/**
	 * @param b
	 * @param a
	 * @return true if such a tuple was new
	 */
	protected abstract boolean primary__index(T b, T a);

	/**
	 * Must set {@link #do__markInferenceAsComplete()} at the right moment. Some implementations do this in the middle
	 * and then re-index all tuples, thus inferring step-by-step.
	 *
	 * @param tupleSink for inferred tuples
	 */
	protected abstract void primary__inference();

	/**
	 * @return true if the primary index is empty
	 */
	protected abstract boolean primary__isEmpty();

	protected final boolean primary__isSymmetric() {
		return this.effectiveSymmetric;
	}

	/**
	 * @return true if this index is transitive
	 */
	protected final boolean primary__isTransitive() {
		return this.effectiveTransitive;
	}

	/**
	 * @param cA
	 * @param infLayer
	 * @return X in (cA,X)
	 */
	protected abstract Iterator<T> primary__query_aX_project_X(Constraint<T> cA, InfLayerRead infLayer);

	/**
	 * @param symmetric
	 * @throws IllegalStateException if the index has an inverse index. This needs to be resolved at higher layers
	 *         before.
	 */
	private final void primary__setSymmetric(final boolean symmetric) throws IllegalStateException {
		/* is there a difference to current effective state? */
		if (symmetric == this.effectiveSymmetric) {
			return;
		}
		// remember change
		this.effectiveSymmetric = symmetric;

		// IMPLEMENT: handle new, changed state
		// if(symmetric) {
		// if(hasInverse()) {
		// // => (this.P sameAs inverse.P), handled at PIndex level
		// throw new IllegalStateException(
		// "Cannot set an index with inverse to symmetric - remove inverse index first");
		// }
		// do__markInferenceAsDirty();
		// // IMPROVE could re-inference now
		// } else {
		// this.clearInf();
		// do__markInferenceAsDirty();
		// // IMPROVE could re-inference now
		// }
	}

	/**
	 * Setting an index or its inverse effectively makes both transitive
	 *
	 * @param transitive
	 */
	private final void primary__setTransitive(final boolean transitive) {
		/* is there a difference to current effective state? */
		if (transitive == primary__isTransitive()) {
			return;
		}

		// remember change
		this.effectiveTransitive = transitive;
		// handle new, changed state
		if (transitive) {
			// activate transitivity, just mark as dirty
			do__markInferenceAsDirty();
			// IMPROVE could re-inference now
		} else {
			// disable transitivity, clear
			this.clearInf();
			do__markInferenceAsDirty();
			// IMPROVE could re-inference now
		}
	}

	/**
	 * @param cA
	 * @param cB
	 * @param infLayer
	 * @return an iterator over all tuples matching (cA,cB) in this index
	 */
	protected abstract Iterator<KeyEntryTuple<T, T>> primary__tupleIterator(Constraint<T> cA, Constraint<T> cB, InfLayerRead infLayer);

	/**
	 * @param infLayer
	 * @return an iterator over all tuples in this index
	 */
	protected abstract Iterator<KeyEntryTuple<T, T>> primary__tuples(InfLayerRead infLayer);

	protected final boolean primary__usesInference() {
		return primary__isSymmetric() || primary__isTransitive();
	}

	@Override
	public final Iterator<T> query_aX_project_X(final Constraint<T> cA, final InfLayerRead infLayer) {
		if (isSecondary()) {
			throw new IllegalStateException("inefficient query");
		} else {
			return primary__query_aX_project_X(cA,infLayer);
		}
	}

	@Override
	public final Iterator<T> query_Xb_project_X(final Constraint<T> cB, final InfLayerRead infLayer) {
		if (isSecondary()) {
			return getInverseIndex().primary__query_aX_project_X(cB,infLayer);
		} else {
			throw new IllegalStateException("inefficient query");
		}
	}

	@Override
	public final void setNoInverseIndexAndClearAll() {
		if (!hasInverse()) {
			return;
		}

		assert hasInverse();
		assert!isSymmetric();

		getInverseIndex().do__setNoInverseIndexAndClearAll();
		do__setNoInverseIndexAndClearAll();
	}

	@Override
	public final boolean setSymmetricFlag(final boolean symmetric) {
		// remember
		final boolean changes = do__setSymmetric(symmetric);
		if (!changes) {
			return false;
		}

		// execute change
		if (isPrimary()) {
			primary__setSymmetric(symmetric);
		} else {
			getInverseIndex().primary__setSymmetric(symmetric);
		}

		return true;
	}

	// inf = dirty, primary = true
	@Override
	public final void setToPrimaryWithInverseIndex(final IInverseTransitiveTupleIndex<T> secondaryInverseIndex) {
		if (log.isDebugEnabled()) {
			log.debug("Set '" + getP() + "'.inverseIndex = '" + secondaryInverseIndex.getP() + "'");
		}
		checkAssertions();

		if (secondaryInverseIndex.getP().equals(getP())) {
			throw new IllegalArgumentException(
					"Cannot set an index as inverse to itself/same P. Use setSymmetric(..) instead.");
		}

		this.inverseIndex = (ITTI_Base<T>) secondaryInverseIndex;
		do__markInferenceAsDirty();
		this.isPrimary = true;
		// double-link from primary to secondary
		secondaryInverseIndex.do__setToSecondaryWithPrimaryInverseIndex(this);

		/* speedup: we infer now before adding from inverse index so that we can incrementally infer while we add */
		primary__inference();

		/* moveFromSecondaryToPrimary: move all tuples from secondary to this one to save space and have a simple
		 * unified view
		 *
		 * While we read from inverseIndex, the inference never reads from it. So we can safely do this. */
		final Iterator<KeyEntryTuple<T, T>> invIt = this.inverseIndex.do__tupleIterator(this.ANY, this.ANY);
		while (invIt.hasNext()) {
			final KeyEntryTuple<T, T> t = invIt.next();
			index(t.getSecond(), t.getFirst());
		}

		this.inverseIndex.primary__clearAll();
		assert isInferenceComplete();
		assert isPrimary();
		checkAssertions();
	}

	@Override
	public final boolean setTransitiveFlag(final boolean transitive) {
		// remember
		final boolean changes = do__setTransitive(transitive);
		if (!changes) {
			return false;
		}

		// execute effect
		if (isPrimary()) {
			primary__setTransitive(transitive);
		} else {
			getInverseIndex().primary__setTransitive(transitive);
		}

		return true;
	}

	@Override
	public final String toString() {
		if (isSecondary()) {
			return getInverseIndex().do__toString();
		} else {
			return do__toString();
		}

	}

	@Override
	public final String toString(final boolean dumpAll) {
		if (dumpAll && isSecondary()) {
			if (log.isInfoEnabled()) {
				return "...see at inverse '" + getInverseIndex().getP() + "'";
			}
		}

		if (isSecondary()) {
			return getInverseIndex().do__toString();
		} else {
			return do__toString();
		}
	}

	@Override
	public final Iterator<KeyEntryTuple<T, T>> tupleIterator(final Constraint<T> cA, final Constraint<T> cB, final InfLayerRead infLayer) {
		if (isSecondary()) {
			return invert(getInverseIndex().primary__tupleIterator(cB, cA,infLayer));
		} else {
			return primary__tupleIterator(cA, cB,infLayer);
		}
	}

	@Override
	public final Iterator<KeyEntryTuple<T, T>> tuples(final InfLayerRead infLayer) {
		if (isSecondary()) {
			return invert(getInverseIndex().primary__tuples(infLayer));
		} else {
			return primary__tuples(infLayer);
		}
	}
}
