package de.xam.tupleinf;

import java.util.Iterator;

import org.xydra.index.IMapSetIndex;
import org.xydra.index.query.Constraint;
import org.xydra.index.query.KeyEntryTuple;

/**
 * Provides a subset of the operations of an {@link IMapSetIndex} (key-value
 * pairs, also called tuples), but with the added capability of
 * computing/managing inverse, symmetric and transitive tuples.
 *
 * The following implications hold about properties:
 *
 * hasInverse <=/=> isSymmetric (never both)
 *
 * isSymmetric ==> isPrimary
 *
 * isTransitive AND hasInverse ==> inverse is also transitive
 *
 * isPrimary AND hasInverse ==> inverse is !primary
 *
 * @author xamde
 *
 * @param <T>
 */
public interface IInverseTransitiveTupleIndex<T> extends ITupleSink<T> {

	/**
	 * Delete all inferred tuples
	 */
	void clearInf();

	/**
	 * Query (?a, ?b)
	 *
	 * Copied from {@link IMapSetIndex}.
	 *
	 * @param cA
	 * @param cB
	 * @param infLayer TODO
	 * @NeverNull
	 * @NeverNull
	 * @return true iff this index contains a tuple matching the given
	 *         constraints.
	 */
	boolean contains(Constraint<T> cA, Constraint<T> cB, InfLayerRead infLayer);

	/**
	 * Query (?a, ?b) for defined a and b
	 *
	 * Copied from {@link IMapSetIndex}.
	 *
	 * @param a
	 * @NeverNull
	 * @param b
	 * @param infLayer
	 * @NeverNull
	 * @return true iff this index contains the tuple
	 */
	boolean contains(T a, T b, InfLayerRead infLayer);

	/**
	 * Copied from {@link IMapSetIndex}.
	 *
	 * Removed a tuple from the index.
	 *
	 * @param a
	 * @NeverNull
	 * @param b
	 * @NeverNull
	 * @return true iff set K contained entry
	 */
	boolean deIndex(T a, T b);

	boolean deIndexInfd(T a, T b);

	/**
	 * Makes this index the secondary index.
	 *
	 * Should be called only from implementations of
	 * {@link #setToPrimaryWithInverseIndex(IInverseTransitiveTupleIndex)}
	 *
	 * @param primaryInverseIndex to be added
	 */
	void do__setToSecondaryWithPrimaryInverseIndex(
			IInverseTransitiveTupleIndex<T> primaryInverseIndex);

	/**
	 * Dump to Xydra Logging as log.info(..)
	 *
	 * @param dumpAll if true, only primary indexes are dumped (which include
	 *            the secondary)
	 */
	void dump(boolean dumpAll);

	/**
	 * @return the inverse index or null @CanBeNull if this index has no inverse
	 *         index. Note that if this index {@link #isSymmetric()}, it is both
	 *         {@link #isPrimary()} and returns null for the inverse index.
	 */
	IInverseTransitiveTupleIndex<T> getInverseIndex();

	/**
	 * @return the P of the index, mostly for debugging
	 */
	T getP();

	/**
	 * Short for {@link #getInverseIndex()} != null
	 *
	 * @return true if this index has an accompanying inverse index.
	 */
	boolean hasInverse();

	/**
	 * Copied from {@link IMapSetIndex}.
	 *
	 * Add a tuple to the index
	 *
	 * @param a
	 * @NeverNull
	 * @param b
	 * @NeverNull
	 * @return true iff set K did not contain entry yet
	 */
	@Override
	boolean index(T a, T b);

	/**
	 * @param a
	 * @param b
	 * @return true if (a,b) wasn't indexed before
	 */
	boolean indexInfd(T a, T b);

	void inferAll();

	/**
	 * @return true if the index is empty
	 */
	boolean isEmpty();

	/**
	 * @return true iff this index: a) has an inverse index and only this index
	 *         is primary; b) has no inverse index
	 */
	boolean isPrimary();

	/**
	 * @return true if this index has an inverse index and the other index is
	 *         the primary index.
	 */
	boolean isSecondary();

	/**
	 * @return true if this index is effectively symmetric, i.e. any tuple (a,b)
	 *         implies the existence of tuple (b,a) in this index.
	 */
	boolean isSymmetric();

	/**
	 * @return true if this index is effectively transitive.
	 */
	boolean isTransitive();

	/**
	 * Query for all ?x in (?a,?x) and project ?x.
	 *
	 * Copied from {@link IMapSetIndex}.
	 *
	 * @param cA
	 * @NeverNull
	 * @return an iterator that ranges over all x where the a in (?a,x) matches
	 *         cA
	 */
	Iterator<T> query_aX_project_X(Constraint<T> cA, InfLayerRead infLayer);

	/**
	 * Query for all ?x in (?x,?b) and project ?x.
	 *
	 * @param cB
	 * @NeverNull
	 * @return an iterator that ranges over all x where the b in (x,?b) matches
	 *         cB
	 */
	Iterator<T> query_Xb_project_X(Constraint<T> cB, InfLayerRead infLayer);

	/**
	 * @param tupleSink for inferred tuples
	 */
	void setInfdTupleSink(ITupleSink<T> tupleSink);

	/**
	 * Remove a binding to an inverse index, if any. This resets the inferred
	 * tuples.
	 */
	void setNoInverseIndexAndClearAll();

	/**
	 * Set this index to symmetric or not.
	 *
	 * Setting an index to symmetric which has an inverse index causes the
	 * inverse index to be merged into this index.
	 *
	 * @param symmetric
	 * @return true if the new value is different from the stored value
	 */
	boolean setSymmetricFlag(boolean symmetric);

	/**
	 * Makes this index the primary index AND the other explicitly the secondary
	 * index by calling
	 * {@link #do__setToSecondaryWithPrimaryInverseIndex(IInverseTransitiveTupleIndex)}
	 *
	 * @param secondaryInverseIndex to be the secondary index.
	 * @throws IllegalArgumentException if this index represent the same P (
	 *             {@link #getP()}) as the inverse index. To achieve this, use
	 *             {@link #setSymmetricFlag(boolean)}.
	 */
	void setToPrimaryWithInverseIndex(IInverseTransitiveTupleIndex<T> secondaryInverseIndex)
			throws IllegalArgumentException;

	/**
	 * If this index is transitive, an inverse index is also transitive. If this
	 * index is set to not transitive, a potential inverse index is also set to
	 * not transitive.
	 *
	 * @param transitive
	 * @return true if the new value is different from the stored value
	 */
	boolean setTransitiveFlag(boolean transitive);

	/**
	 * @param dumpAll
	 * @return same as {@link #dump(boolean)}
	 */
	String toString(boolean dumpAll);

	/**
	 * Query(?a,?b)
	 *
	 * Copied from {@link IMapSetIndex}.
	 *
	 * Contains inverse tuples only iff this tuple index {@link #isSymmetric()}
	 *
	 * @param cA constraint on the key @NeverNull
	 * @param cB constraint on the value @NeverNull
	 * @return an iterator over all result tuples matching the constraints
	 */
	Iterator<KeyEntryTuple<T, T>> tupleIterator(Constraint<T> cA, Constraint<T> cB, InfLayerRead infLayer);

	/**
	 * @return same as {@link #tupleIterator(Constraint, Constraint)} if both constraint are a wild-card
	 */
	Iterator<KeyEntryTuple<T, T>> tuples(InfLayerRead infLayer);
}
