package de.xam.triplerules.impl;

import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;

import org.xydra.log.api.Logger;
import org.xydra.log.api.LoggerFactory;

import de.xam.triplerules.IRuleAction;
import de.xam.triplerules.ITriplePattern;
import de.xam.triplerules.ITripleRule;

public class TripleRule<K, L, M> implements ITripleRule<K, L, M> {

	private static final Logger log = LoggerFactory.getLogger(TripleRule.class);

	private final IRuleAction<K, L, M> action = new RuleAction<K, L, M>();

	private final RuleCondition<K, L, M> condition = new RuleCondition<K, L, M>();

	private final String label;

	/**
	 * A rule with a unique label
	 *
	 * @param ruleLabel used for hashCode & equals
	 */
	public TripleRule(final String ruleLabel) {
		this.label = ruleLabel;
	}

	@Override
	public RuleCondition<K, L, M> condition() {
		return this.condition;
	}

	@Override
	public IRuleAction<K, L, M> action() {
		return this.action;
	}

	@Override
	public String dump() {
		log.info(this.toString());
		return "";
	}

	@Override
	public String toString() {
		final StringBuffer buf = new StringBuffer();
		buf.append("/* RULE " + this.label + " */\n");
		buf.append(this.condition.toString());
		for (final ITriplePattern<K, L, M> action : action().patterns()) {
			buf.append(" THEN " + action.toString() + "\n");
		}
		return buf.toString();

	}

	private final Map<String, Integer> varname2id = new HashMap<String, Integer>();

	private int varNumber = 0;

	@Override
	public int translateNameToId(final String name) {
		Integer i = this.varname2id.get(name);
		if (i == null) {
			final int j = this.varNumber++;
			i = j;
			this.varname2id.put(name, i);
		}
		return i;
	}

	@Override
	public void compile() {
		for (final ITriplePattern<K, L, M> pattern : this.condition.patterns()) {
			pattern.compile(this);
		}
		for (final ITriplePattern<K, L, M> pattern : this.action.patterns()) {
			pattern.compile(this);
		}
	}

	@Override
	public int varCount() {
		return this.varNumber;
	}

	@Override
	public String label() {
		return this.label;
	}

	@Override
	public String translateIdToName(final int i) {
		for (final Entry<String, Integer> e : this.varname2id.entrySet()) {
			if (e.getValue() == i) {
				return e.getKey();
			}
		}
		return null;
	}

	@Override
	public int hashCode() {
		return this.label.hashCode();
	}

	@SuppressWarnings("rawtypes")
	@Override
	public boolean equals(final Object other) {
		return other instanceof ITripleRule && ((ITripleRule) other).label().equals(label());
	}
}
