package de.xam.triplerules.impl;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.TreeMap;

import org.xydra.core.util.Clock;
import org.xydra.index.IEntrySet;
import org.xydra.index.IMapSetIndex;
import org.xydra.index.ITripleIndex;
import org.xydra.index.impl.FastEntrySetFactory;
import org.xydra.index.impl.MapSetIndex;
import org.xydra.index.iterator.ITransformer;
import org.xydra.index.iterator.Iterators;
import org.xydra.index.query.ITriple;
import org.xydra.index.query.KeyEntryTuple;
import org.xydra.index.query.KeyKeyEntryTuple;
import org.xydra.index.query.Wildcard;
import org.xydra.log.api.Logger;
import org.xydra.log.api.LoggerFactory;

import de.xam.texthtml.text.HumanReadableText;
import de.xam.triplerules.IRuleEngine;
import de.xam.triplerules.ITriplePattern;
import de.xam.triplerules.ITripleRule;
import de.xam.triplerules.TripleSetWithDirtyFlag;

/**
 * RuleEngine manages the rules themselves; All inference work is done in the {@link InferenceEngine} via static
 * methods.
 *
 *
 * INFO : Processed 134900 triples. Inferred 263052 triples at de.xam.triplerules.impl.RuleEngine.(RuleEngine.java:230)
 * on 2013-11-18 18:43:15,701
 *
 * Exception in thread "main" java.lang.OutOfMemoryError: Java heap space
 *
 * @author xamde
 *
 * @param <K>
 * @param <L>
 * @param <M>
 */
public class RuleEngine<K extends Serializable, L extends Serializable, M extends Serializable>
		implements IRuleEngine<K, L, M>, IRuleManager<K, L, M> {

	static class InfStats {
		int iterations = 0;
		long newTriples = 0;

		public InfStats(final int iterations, final long newTriples) {
			this.iterations = iterations;
			this.newTriples = newTriples;
		}

		@Override
		public String toString() {
			return this.iterations + " iterations, " + this.newTriples + " new triples";
		}
	}

	@SuppressWarnings("unused")
	private static boolean HACK_MODE_SLOW = false;

	// use logger on {@link de.xam.triplerules.impl.InfLog} to see a reasoning trace
	private static final Logger log = LoggerFactory.getLogger(RuleEngine.class);

	/** updated at runtime */
	private Set<ITripleRule<K, L, M>> activeRules = new HashSet<ITripleRule<K, L, M>>();

	private ICostEstimator<K, L, M> costEstimator;

	private int inferredTriples = 0;

	private TripleSetWithDirtyFlag<K, L, M> lowLevelInferenceBuffer;

	/** updated at runtime */
	private Set<ITripleRule<K, L, M>> matchedRules = new HashSet<ITripleRule<K, L, M>>();

	private int processedInputTriples = 0;

	private long reportingThreshold = 1000;

	/** is compiled once */
	private final MapSetIndex<ITripleRule<K, L, M>, ITripleRule<K, L, M>> ruleDependencies = new MapSetIndex<ITripleRule<K, L, M>, ITripleRule<K, L, M>>(
			new FastEntrySetFactory<ITripleRule<K, L, M>>());

	/** state */
	private final Set<ITripleRule<K, L, M>> rules = new HashSet<ITripleRule<K, L, M>>();

	private boolean rulesAreCompiled = false;

	public RuleEngine() {
		this(new DefaultCostEstimator<K, L, M>());
	}

	public RuleEngine(final ICostEstimator<K, L, M> costEstimator) {
		this.costEstimator = costEstimator;
	}

	/**
	 * Infer from a rule that has matched the transitive set of rules that might match in the next inference iteration.
	 *
	 * @param matchedRule
	 */
	private void activateRules(final ITripleRule<K, L, M> matchedRule) {
		final IEntrySet<ITripleRule<K, L, M>> otherRules = this.ruleDependencies.lookup(matchedRule);
		if (otherRules != null) {
			for (final ITripleRule<K, L, M> other : otherRules) {
				if (!this.activeRules.contains(matchedRule)) {
					this.activeRules.add(other);
					activateRules(other);
				}
			}
		}
	}

	@Override
	public void addRule(final ITripleRule<K, L, M> rule) {
		this.rules.add(rule);
		this.rulesAreCompiled = false;
	}

	/**
	 * Great if source == target
	 *
	 * @param source
	 * @param target
	 * @return
	 */
	private ITripleIndex<K, L, M> combine(final ITripleIndex<K, L, M> source, final ITripleIndex<K, L, M> target) {
		ITripleIndex<K, L, M> combined;
		if (source == target) {
			combined = source;
		} else {
			combined = CombinedTripleIndex.create(source, target);
		}
		return combined;
	}

	private void compileRules() {
		for (final ITripleRule<K, L, M> rule : this.rules) {
			rule.compile();
			// index rule dependencies
			for (final ITripleRule<K, L, M> otherRule : this.rules) {
				// a rule could trigger itself
				if (RuleUtils.couldTrigger(rule, otherRule)) {
					this.ruleDependencies.index(rule, otherRule);
				}
			}

		}

		if (log.isDebugEnabled()) {
			final Iterator<KeyEntryTuple<ITripleRule<K, L, M>, ITripleRule<K, L, M>>> it = this.ruleDependencies
					.tupleIterator(new Wildcard<ITripleRule<K, L, M>>(), new Wildcard<ITripleRule<K, L, M>>());
			while (it.hasNext()) {
				final KeyEntryTuple<ITripleRule<K, L, M>, ITripleRule<K, L, M>> tuple = it.next();
				log.debug("Rule triggers " + tuple.getFirst().label() + " -> " + tuple.getSecond().label());
			}
		}

		this.rulesAreCompiled = true;
	}

	public void dumpRules() {
		final Map<String, ITripleRule<K, L, M>> map = new TreeMap<>();
		for (final ITripleRule<K, L, M> rule : this.rules) {
			map.put(rule.label(), rule);
		}
		final StringBuilder b = new StringBuilder();
		for (final Entry<String, ITripleRule<K, L, M>> e : map.entrySet()) {
			b.append(e.getValue().toString());
		}
		log.info("\n" + b.toString());
	}

	/**
	 * Set consecutive variables IDs instead of names.
	 */
	private synchronized void ensureCompiledRules() {
		if (this.rulesAreCompiled) {
			return;
		}
		compileRules();
	}

	/**
	 * This method exists only for debugging, statistics etc. It is not required for usage of {@link RuleEngine}.
	 *
	 * @return a set of tuples of rules which depend on each other. I.e. a tuple (A,B) means that after rule A fired,
	 *         rule B might fire, too.
	 */
	public IMapSetIndex<ITripleRule<K, L, M>, ITripleRule<K, L, M>> getRuleDependencies() {
		ensureCompiledRules();
		return this.ruleDependencies;
	}

	/**
	 * Callers should not add or change the rules.
	 *
	 * This method exists only for debugging, statistics etc. It is not required for usage of {@link RuleEngine}.
	 *
	 * @return the internal set of rules
	 */
	public Set<ITripleRule<K, L, M>> getRules() {
		return this.rules;
	}

	/**
	 * Compile rules; infer first iteration in a special way: For each rule, choose the most specific pattern and query
	 * the tripleIndex with it. Use the obtained matches to create a binding and try to match the resulting refined
	 * other patterns of the rule condition as well. All new inferred triples go into a temporary list. Then this list
	 * is also copied into the tripleIndex.
	 *
	 * The second and all further iterations run like this:
	 *
	 * For each active rule: For each incoming triple: Check if any pattern of the rule matches the new triple, and if
	 * so, check if all other patterns can be matched with the obtained binding from the triple index.
	 *
	 * @see de.xam.triplerules.IRuleEngine#inferAll(org.xydra.index.ITripleIndex, org.xydra.index.ITripleIndex)
	 */
	@Override
	public void inferAll(final ITripleIndex<K, L, M> sourceTripleIndex, final ITripleIndex<K, L, M> targetTripleIndex) {
		ensureCompiledRules();

		log.info("[0=iteration.x=ruleNumber] ==================== Initial iteration from all triples...");
		final Set<ITriple<K, L, M>> startInfTripleSet = new HashSet<ITriple<K, L, M>>();
		int ruleNo = 1;
		/* For each rule */
		for (final ITripleRule<K, L, M> rule : this.rules) {
			if (log.isDebugEnabled()) {
				log.debug("[0." + ruleNo + "=rule] Processing rule '" + rule.label() + "' (Rule " + ruleNo + " of "
						+ this.rules.size() + ")");
			}

			final ITriplePattern<K, L, M> bestFirstPattern = InferenceEngine.getCheapestPattern(rule.condition(),
					this.costEstimator);
			if (log.isDebugEnabled()) {
				log.debug("[0." + ruleNo + "]" + " Cheapest pattern: " + bestFirstPattern);
			}
			final Iterator<? extends ITriple<K, L, M>> triples = InferenceEngine.query(rule, bestFirstPattern,
					sourceTripleIndex);
			final int thisRuleNo = ruleNo;
			final Iterator<ConditionBinding<K, L, M>> bindings = Iterators.transform(triples,
					new ITransformer<ITriple<K, L, M>, ConditionBinding<K, L, M>>() {

						@Override
						public ConditionBinding<K, L, M> transform(final ITriple<K, L, M> triple) {
							log.trace("[0." + thisRuleNo + ".triple] " + triple);
							final ConditionBinding<K, L, M> binding = ConditionBinding.createInitialBinding(rule,
									bestFirstPattern, triple);
							log.trace("[0." + thisRuleNo + ".triple] binding " + binding);
							return binding;
						}
					});

			final Iterator<ConditionBinding<K, L, M>> explodedBindings = InferenceEngine
					.explodeBindingsRecursively(rule, sourceTripleIndex, bindings, this.costEstimator);

			if (explodedBindings.hasNext()) {
				log.debug("Rule '" + rule.label() + "' matched.");
				markRuleAsMatched(rule);
			}

			while (explodedBindings.hasNext()) {
				final ConditionBinding<K, L, M> explodedBinding = explodedBindings.next();

				assert explodedBinding.size() == explodedBinding.capacity() : "bindingSize=" + explodedBinding.size()
						+ " bindingCapacity=" + explodedBinding.capacity();

				RuleUtils.materialiseTriples(rule, explodedBinding, targetTripleIndex, startInfTripleSet);
			}
			ruleNo++;
		}

		log.info("==================== Done: Initial iteration from all triples. Inferred " + startInfTripleSet.size()
				+ " triples");

		final ITripleIndex<K, L, M> combined = combine(sourceTripleIndex, targetTripleIndex);
		// activate all rules
		this.activeRules.addAll(this.rules);

		// iteration 1..n
		final InfStats infStats = inferLoop(startInfTripleSet, combined, targetTripleIndex);
		log.info("==================== Done: Iteration from all triples. " + infStats);
	}

	/**
	 * @param source
	 * @param s @NeverNull
	 * @param p @NeverNull
	 * @param o @NeverNull
	 * @param target @NeverNull
	 */
	public void inferIncrementalFromTriple(final ITripleIndex<K, L, M> source, final K s, final L p, final M o,
			final ITripleIndex<K, L, M> target) {
		if (log.isDebugEnabled()) {
			log.debug("(" + s + "," + p + "," + o + ") infer incremental from here...");
		}

		ensureCompiledRules();

		final ITripleIndex<K, L, M> combined = combine(source, target);
		// activate all rules
		this.activeRules.addAll(this.rules);

		final Collection<ITriple<K, L, M>> startInf = new HashSet<ITriple<K, L, M>>();
		startInf.add(new KeyKeyEntryTuple<K, L, M>(s, p, o));
		inferLoop(startInf, combined, target);
	}

	/**
	 * @param startInf
	 * @param sourceTripleIndex
	 * @param targetTripleIndex
	 */
	private InfStats inferLoop(final Collection<ITriple<K, L, M>> startInf,
			final ITripleIndex<K, L, M> sourceTripleIndex, final ITripleIndex<K, L, M> targetTripleIndex) {
		int iteration = 1;
		Collection<ITriple<K, L, M>> triplesInferredInOneIteration = startInf;
		long newTriples = 0;

		/* while either newInf (filled by RuleEngine) or lowLevelInferenceBuffer (filled by smart triple index) are
		 * producing new triples */
		while (!triplesInferredInOneIteration.isEmpty() || !isLowLevelInferenceEmpty()) {

			// commit this round's new inferences to target
			for (final ITriple<K, L, M> triple : triplesInferredInOneIteration) {
				targetTripleIndex.index(triple.getKey1(), triple.getKey2(), triple.getEntry());
			}

			// respect optional low level inf engine results
			if (!isLowLevelInferenceEmpty()) {
				log.debug("Using also " + this.lowLevelInferenceBuffer.size() + " indirectly inferred triples");
				// DEBUG
				if (log.isTraceEnabled() && this.lowLevelInferenceBuffer.size() <= 10) {
					final List<ITriple<K, L, M>> list = new ArrayList<ITriple<K, L, M>>();
					this.lowLevelInferenceBuffer.exportAllTo(list);
					for (final ITriple<K, L, M> tr : list) {
						log.trace("LowLevel: " + tr);
					}
				}
				this.lowLevelInferenceBuffer.exportAllTo(triplesInferredInOneIteration);
				this.lowLevelInferenceBuffer.clear();
			}

			// switch
			final Collection<ITriple<K, L, M>> lastInf = triplesInferredInOneIteration;
			newTriples += lastInf.size();

			// compute next iteration
			log.trace("==================== Infer iteration #" + iteration + "; lastInf.size=" + lastInf.size());
			triplesInferredInOneIteration = inferOneIteration(lastInf, sourceTripleIndex, iteration,
					this.costEstimator);
			iteration++;
		}
		assert triplesInferredInOneIteration.isEmpty() : "newInf is empty, nothing to commit";

		// stats
		final InfStats infStats = new InfStats(iteration, newTriples);
		if (log.isTraceEnabled()) {
			log.trace("==================== Infered " + infStats.toString());
		}
		return infStats;
	}

	/**
	 * From all new triples inferred in last iterations: Try to infer new triples for the next iteration.
	 *
	 * @param lastInf inferences created in last iteration; new inferences are only triggered by these.
	 * @param tripleIndex where to look up existing tuples, including those inferred during previous iterations
	 * @param iteration
	 * @param costEstimator
	 *
	 * @return new inferences
	 */
	private Collection<ITriple<K, L, M>> inferOneIteration(final Collection<? extends ITriple<K, L, M>> lastInf,
			final ITripleIndex<K, L, M> tripleIndex, final int iteration, final ICostEstimator<K, L, M> costEstimator) {

		if (log.isDebugEnabled()) {
			for (final ITripleRule<K, L, M> rule : this.matchedRules) {
				log.debug("### Rule matched: " + rule.label());
			}

			final List<ITripleRule<K, L, M>> activeRulesList = Iterators.toList(this.activeRules.iterator());
			Collections.sort(activeRulesList, new Comparator<ITripleRule<K, L, M>>() {

				@Override
				public int compare(final ITripleRule<K, L, M> o1, final ITripleRule<K, L, M> o2) {
					return o1.label().compareTo(o2.label());
				}
			});

			for (final ITripleRule<K, L, M> rule : activeRulesList) {
				log.debug("### Active rule: " + rule.label());
			}
		}

		if (log.isDebugEnabled()) {
			final Iterator<ITriple<K, L, M>> it = tripleIndex.getTriples(new Wildcard<K>(), new Wildcard<L>(),
					new Wildcard<M>());
			final int count = Iterators.count(it);
			log.debug("Triple base contains " + HumanReadableText.largeNumber(count) + " triples. Last inf = "
					+ HumanReadableText.largeNumber(lastInf.size()) + " triples.");
		}

		final Collection<ITriple<K, L, M>> newInf = new HashSet<ITriple<K, L, M>>();

		// for each rule: inspect all newly inferred triples
		for (final ITripleRule<K, L, M> rule : this.rules) {
			if (!this.activeRules.contains(rule)) {
				log.debug("There cannot be a match for rule " + rule.label() + "");
				continue;
			}

			// run rule
			inferOneIteration_OneRule(rule, lastInf, tripleIndex, iteration, newInf, costEstimator);
		}

		// compute which rules can match next
		this.activeRules = new HashSet<ITripleRule<K, L, M>>();
		for (final ITripleRule<K, L, M> rule : this.matchedRules) {
			activateRules(rule);
		}
		this.matchedRules = new HashSet<ITripleRule<K, L, M>>();

		return newInf;
	}

	/**
	 * @param rule
	 * @param lastInf
	 * @param tripleIndex
	 * @param iteration
	 * @param newInf is filled with new triples
	 * @param costEstimator
	 */
	private void inferOneIteration_OneRule(final ITripleRule<K, L, M> rule,
			final Collection<? extends ITriple<K, L, M>> lastInf, final ITripleIndex<K, L, M> tripleIndex,
			final int iteration, final Collection<ITriple<K, L, M>> newInf,
			final ICostEstimator<K, L, M> costEstimator) {
		if (log.isDebugEnabled()) {
			log.debug("> Running rule '" + rule.label() + "' on " + lastInf.size() + " triples                     ");
		}

		// stats
		int inferredByThisRule = 0;
		final Clock clock = new Clock().start();

		// for each triple
		for (final ITriple<K, L, M> triple : lastInf) {
			this.processedInputTriples++;

			// measure performance
			if (this.processedInputTriples % this.reportingThreshold == 0) {
				this.reportingThreshold *= 2;
				log.info(">> Processed " + HumanReadableText.largeNumber(this.processedInputTriples)
						+ " input triples. Inferred  " + HumanReadableText.largeNumber(this.inferredTriples)
						+ " triples");
				final long millis = clock.stopAndGetDuration("triples");
				clock.start();
				// // FIXME hack: adaptive log levels if things get slow
				// if(!HACK_MODE_SLOW && millis > 1000) {
				// Log4jUtils.setLevel(InferenceEngine.class, Level.DEBUG);
				// Log4jUtils.setLevel(RuleEngine.class, Level.DEBUG);
				// Log4jUtils.setLevel(RuleUtils.class, Level.DEBUG);
				// HACK_MODE_SLOW = true;
				// }
				// if(HACK_MODE_SLOW && millis < 100) {
				// Log4jUtils.setLevel(InferenceEngine.class, Level.INFO);
				// Log4jUtils.setLevel(RuleEngine.class, Level.INFO);
				// Log4jUtils.setLevel(RuleUtils.class, Level.INFO);
				// HACK_MODE_SLOW = false;
				// }
				log.debug("Millis = " + millis);
			}

			if (log.isDebugEnabled()) {
				log.debug(">> Processing triple " + triple);
			}

			inferredByThisRule += InferenceEngine.inferOneIteration_OneRule_OneTriple(this, rule, triple, tripleIndex,
					newInf, iteration, costEstimator);
		}

		this.inferredTriples += inferredByThisRule;

		if (log.isDebugEnabled() && inferredByThisRule > 0) {
			log.debug("> Rule '" + rule.label() + "' inferred " + inferredByThisRule + " triples");
		}
		if (log.isDebugEnabled() && newInf.size() > 0 && iteration > 1) {
			log.debug("> Total NewInf.size = " + newInf.size());
		}
	}

	private boolean isLowLevelInferenceEmpty() {
		return this.lowLevelInferenceBuffer == null || this.lowLevelInferenceBuffer.isEmpty();
	}

	@Override
	public void markRuleAsMatched(final ITripleRule<K, L, M> rule) {
		this.matchedRules.add(rule);
	}

	@Override
	public void setCostEstimator(final ICostEstimator<K, L, M> costEstimator) {
		this.costEstimator = costEstimator;
	}

	public void setLowLevelTripleInferenceSource(final TripleSetWithDirtyFlag<K, L, M> lowLevelInferenceBuffer) {
		this.lowLevelInferenceBuffer = lowLevelInferenceBuffer;
	}

}
