package de.xam.triplerules.impl;

import java.util.Iterator;

import org.xydra.index.ITripleIndex;
import org.xydra.index.IndexUtils;
import org.xydra.index.iterator.Iterators;
import org.xydra.index.query.Constraint;
import org.xydra.index.query.ITriple;
import org.xydra.log.api.Logger;
import org.xydra.log.api.LoggerFactory;

/**
 * Reads go to primary and secondary layer; write affect only secondary. So deleting some existing triple might have no
 * effect if the triple was in the primary layer.
 *
 * @author xamde
 *
 * @param <K>
 * @param <L>
 * @param <M>
 */
public class CombinedTripleIndex<K, L, M> implements ITripleIndex<K, L, M> {

	private static final Logger log = LoggerFactory.getLogger(CombinedTripleIndex.class);

	public static <K, L, M> CombinedTripleIndex<K, L, M> create(final ITripleIndex<K, L, M> primary,
			final ITripleIndex<K, L, M> secondary) {
		return new CombinedTripleIndex<K, L, M>(primary, secondary);
	}

	public CombinedTripleIndex(final ITripleIndex<K, L, M> primary, final ITripleIndex<K, L, M> secondary) {
		super();
		this.primary = primary;
		this.secondary = secondary;
	}

	private final ITripleIndex<K, L, M> primary, secondary;

	@Override
	public void clear() {
		this.secondary.clear();
	}

	@Override
	public boolean isEmpty() {
		return this.primary.isEmpty();
	}

	@Override
	public boolean contains(final Constraint<K> c1, final Constraint<L> c2, final Constraint<M> c3) {
		return this.primary.contains(c1, c2, c3) || this.secondary.contains(c1, c2, c3);
	}

	@Override
	public boolean contains(final K s, final L p, final M o) {
		return this.primary.contains(s, p, o) || this.secondary.contains(s, p, o);
	}

	@Override
	public boolean deIndex(final K s, final L p, final M o) {
		return this.secondary.deIndex(s, p, o);
	}

	@Override
	public String dump() {
		log.info("=== Primary ===");
		this.primary.dump();
		log.info("=== Secondary ===");
		this.secondary.dump();
		return "";
	}

	@Override
	public boolean index(final K s, final L p, final M o) {
		if (this.primary.contains(s, p, o)) {
			return false;
		}

		return this.secondary.index(s, p, o);
	}

	@Override
	public Iterator<ITriple<K, L, M>> getTriples(final Constraint<K> c1, final Constraint<L> c2,
			final Constraint<M> c3) {
		if (log.isTraceEnabled()) {
			log.trace("--- primary.query" + IndexUtils.asQuery(c1, c2, c3) + " from "
					+ this.primary.getClass().getName());
			Iterator<? extends ITriple<K, L, M>> it = this.primary.getTriples(c1, c2, c3);
			while (it.hasNext()) {
				final ITriple<K, L, M> t = it.next();
				log.trace("- triple: " + t);
			}
			log.trace("--- secondary.query" + IndexUtils.asQuery(c1, c2, c3) + " from "
					+ this.secondary.getClass().getName());
			it = this.secondary.getTriples(c1, c2, c3);
			while (it.hasNext()) {
				final ITriple<K, L, M> t = it.next();
				log.trace("- triple: " + t);
			}
		}

		return Iterators.concat(this.primary.getTriples(c1, c2, c3), this.secondary.getTriples(c1, c2, c3));
	}

	@Override
	public Iterator<ITriple<K, L, M>> getTriples(final K s, final L p, final M o) {
		if (log.isTraceEnabled()) {
			log.trace("--- primary.query" + IndexUtils.asQuery(s, p, o) + " from " + this.primary.getClass().getName());
			Iterator<? extends ITriple<K, L, M>> it = this.primary.getTriples(s, p, o);
			while (it.hasNext()) {
				final ITriple<K, L, M> t = it.next();
				log.trace("- triple: " + t);
			}
			log.trace("--- secondary.query" + IndexUtils.asQuery(s, p, o) + " from "
					+ this.secondary.getClass().getName());
			it = this.secondary.getTriples(s, p, o);
			while (it.hasNext()) {
				final ITriple<K, L, M> t = it.next();
				log.trace("- triple: " + t);
			}
		}

		return Iterators.concat(this.primary.getTriples(s, p, o), this.secondary.getTriples(s, p, o));
	}

	@Override
	public Iterator<M> getObjects_XXX() {
		return Iterators.distinct(Iterators.concat(this.primary.getObjects_XXX(), this.secondary.getObjects_XXX()));
	}

	@Override
	public Iterator<M> getObjects_SPX(final K s, final L p) {
		return Iterators
				.distinct(Iterators.concat(this.primary.getObjects_SPX(s, p), this.secondary.getObjects_SPX(s, p)));
	}

	@Override
	public Iterator<L> getPredicates_XXX() {
		return Iterators.distinct(Iterators.concat(this.primary.getPredicates_XXX(), this.secondary.getPredicates_XXX()));
	}

	@Override
	public Iterator<L> getPredicates_SXX(final K s) {
		return Iterators
				.distinct(Iterators.concat(this.primary.getPredicates_SXX(s), this.secondary.getPredicates_SXX(s)));
	}

	@Override
	public Iterator<L> getPredicates_SXO(final K s, final M o) {
		return Iterators.distinct(
				Iterators.concat(this.primary.getPredicates_SXO(s, o), this.secondary.getPredicates_SXO(s, o)));
	}

	@Override
	public Iterator<K> getSubjects_XXX() {
		return Iterators.distinct(Iterators.concat(this.primary.getSubjects_XXX(), this.secondary.getSubjects_XXX()));
	}

	@Override
	public Iterator<K> getSubjects_XPO(final L p, final M o) {
		return Iterators
				.distinct(Iterators.concat(this.primary.getSubjects_XPO(p, o), this.secondary.getSubjects_XPO(p, o)));
	}

	// @Override
	// public IMapMapSetDiff<K, L, M> computeDiff(final ITripleIndex<K, L, M> other) {
	// throw new UnsupportedOperationException();
	// }

}
