package de.xam.textsearch.score;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.xydra.index.impl.MapIndex;
import org.xydra.index.iterator.Iterators;
import org.xydra.log.api.Logger;
import org.xydra.log.api.LoggerFactory;

import de.xam.textsearch.Match;
import de.xam.textsearch.spi.IContentResolver;
import de.xam.textsearch.text.ITextQuery;

/**
 * Utilities for scoring, independent of {@link IScorer}
 *
 * @author xamde
 * @param <V>
 */
public class Scores<V> {

	private static final Logger log = LoggerFactory.getLogger(Scores.class);

	private final IContentResolver<V> contentResolver;

	public Scores(final IContentResolver<V> contentResolver) {
		super();
		this.contentResolver = contentResolver;
	}

	public IScorer<V> getScorerFulltextDefault() {
		return new DefaultScorer<V>();
	}

	/**
	 * @param textquery
	 * @param resultIt
	 * @param scorer
	 * @return sorted, scored matches
	 */
	public List<Match<V>> score(final ITextQuery<V> textquery, final Iterator<V> resultIt, final IScorer<V> scorer) {
		/* an ID can be indexed for several fragments ... TODO doc more */
		final MapIndex<V, Match<V>> map = new MapIndex<V, Match<V>>();

		while (resultIt.hasNext()) {
			final V id = resultIt.next();
			final Match<V> match = new Match<V>(id);
			final String content = this.contentResolver.getContentAsStringForTextIndexing(id);

			if (content == null) {
				/* we read a stale index and just remove this ID from search result */
				log.warn("Read stale entity from index with ID=[" + id + "]");
				continue;
			}

			scorer.score(match, content, textquery.getPhrase(), textquery.getWords(),
					textquery.getFragments());

			// TODO why this? document
			final Match<V> existing = map.lookup(id);
			if (existing == null) {
				map.index(id, match);
			} else {
				// FIXME KILL
				log.warn("For ID='"+id+"' have two matches 1="+existing+" 2="+match);

				if (existing.q() >= match.q()) {
					// keep existing
				} else {
					map.index(id, match);
				}
			}
		}

		// copy over
		return Iterators.toList(map.iterator());
	}

	/**
	 * @param textquery
	 * @param resultSet
	 * @param scorer
	 * @return sorted, scored matches
	 */
	public List<Match<V>> score(final ITextQuery<V> textquery, final Set<V> resultSet, final IScorer<V> scorer) {
		final List<Match<V>> matches = new ArrayList<Match<V>>(resultSet.size());
		for (final V id : resultSet) {
			final Match<V> match = new Match<V>(id);
			final String content = this.contentResolver.getContentAsStringForTextIndexing(id);
			scorer.score(match, content, textquery.getPhrase(), textquery.getWords(), textquery.getFragments());
			matches.add(match);
		}
		return matches;
	}

}
