package de.xam.googleanalytics.tracker;

import de.xam.googleanalytics.UrlAndHashUtils;

/**
 * Expires never.
 *
 * <code>__utma='domainhash.randomValue.ftime.ltime.stime.2</code>
 *
 * Format:
 *
 * 'domainhash.unique.ftime.ltime.stime.sessioncount' where:
 *
 * <pre>
 * domainhash = hash of the domain name of the website
 * unique = a randomly generated 31 bit integer
 * ftime = UTC timestamp of first visitor session
 * ltime = UTC timestamp of last visitor session
 * stime = UTC timestamp of current visitor session
 * sessioncount = number of sessions; always incremented for each new session
 * </pre>
 *
 * @author xamde
 */
public class GaUtmaCookie extends GaUtmbCookie {

	/**
	 * State must be set via {@link #setFromCookieString(String)}
	 *
	 * @param domainName
	 * @param domainHash
	 */
	public GaUtmaCookie(final String domainName, final String domainHash) {
		super(domainName, domainHash);
	}

	public GaUtmaCookie(final String cookieString) {
		setFromCookieString(cookieString);
	}

	public GaUtmaCookie(final String domainName, final long the31BitId, final long firstVisitStartTime,
			final long lastVisitStartTime, final long currentSessionStartTime, final long sessionCount) {
		super(domainName, null, currentSessionStartTime, sessionCount);
		this.the31BitId = the31BitId;
		this.firstVisitStartTime = firstVisitStartTime;
		this.lastVisitStartTime = lastVisitStartTime;
	}

	@Override
	public String toCookieString() {
		return getDomainHash() + "." + this.the31BitId + "." + this.firstVisitStartTime + "."
				+ this.lastVisitStartTime + "." + this.currentSessionStartTime + "."
				+ this.sessionCount;
	}

	/**
	 * @param cookieString
	 *            format: (domainHash).(visitorId).(firstVisit).(previousSession
	 *            ).(currentSession).(sessionNumber)
	 */
	@Override
	public void setFromCookieString(final String cookieString) throws IllegalArgumentException {
		// try to parse
		final String[] dotParts = cookieString.split("\\.");
		if (dotParts.length == 6) {
			this.domainHash = dotParts[0];
			this.the31BitId = UrlAndHashUtils.parseAsLong(dotParts[1]);
			this.firstVisitStartTime = UrlAndHashUtils.parseAsLong(dotParts[2]);
			this.lastVisitStartTime = UrlAndHashUtils.parseAsLong(dotParts[3]);
			this.currentSessionStartTime = UrlAndHashUtils.parseAsLong(dotParts[4]);
			this.sessionCount = UrlAndHashUtils.parseAsLong(dotParts[5]);
		} else {
			throw new IllegalArgumentException("Could not parse '" + cookieString
					+ "' into six dot-separated parts");
		}
	}

	public long getUniqueId() {
		return this.the31BitId;
	}

	public long getFirstVisit() {
		return this.firstVisitStartTime;
	}

	/**
	 * UTC time-stamp of first visitor session <em>in seconds</em>
	 */
	public long firstVisitStartTime;

	/**
	 * UTC timestamp of last visitor session <em>in seconds</em>
	 */
	public long lastVisitStartTime;

	public long the31BitId;

}
