package com.calpano.common.shared.xydrautils.field;

import org.xydra.base.BaseRuntime;
import org.xydra.base.XId;
import org.xydra.base.rmof.XReadableField;
import org.xydra.base.rmof.XReadableObject;
import org.xydra.base.rmof.XWritableField;
import org.xydra.base.rmof.XWritableObject;
import org.xydra.base.value.XV;
import org.xydra.base.value.XValue;
import org.xydra.core.model.XField;
import org.xydra.core.model.XObject;

/**
 * An extensible accessor for XValues stored in an {@link XField} of an
 * {@link XObject}.
 *
 * @author xamde
 *
 * @param <J>
 *            any java type
 * @param <V>
 *            a Xydra-supported {@link XValue} type
 */
public class ExtensibleFieldProperty<J, V extends XValue> {

	/**
	 * Can convert from foreign type to supported type
	 *
	 * @author xamde
	 *
	 * @param <J>
	 *            supported java type
	 * @param <V>
	 *            Xydra value type
	 */
	public static interface Converter<J, V extends XValue> {
		/**
		 * @param xydraType
		 *            might be null
		 * @return javaType
		 */
		J toJavaType(V xydraType);

		/**
		 * @param anyType
		 *            might be null
		 * @return a supported Xydra type
		 */
		V toXydraValue(J anyType);
	}

	private final Converter<J, V> converter;
	protected XId fieldId;

	/**
	 * @param fieldIdString
	 *            e.g. 'name' or 'email'
	 * @param converter
	 *            must be able to convert from Xydra value type to Java type.
	 */
	public ExtensibleFieldProperty(final String fieldIdString, final Converter<J, V> converter) {
		this.fieldId = BaseRuntime.getIDProvider().fromString(fieldIdString);
		this.converter = converter;
	}

	/**
	 * @param xo
	 *            from which to read
	 * @return the stored value of the field as an instance of type <T>
	 */
	@SuppressWarnings("unchecked")
	public J getValue(final XReadableObject xo) {
		assert xo != null : "xo was null";
		assert this.fieldId != null;

		final XReadableField xf = xo.getField(this.fieldId);
		if (xf == null) {
			return this.converter.toJavaType(null);
		}

		V specificXValue = null;
		final XValue xvalue = xf.getValue();

		if (xvalue != null) {
			if (xvalue.equals(XV.toValue("_NoValueReadCache"))) {
				throw new AssertionError();
			}
			try {
				specificXValue = (V) xvalue;
			} catch (final ClassCastException e) {
				throw new RuntimeException("Stored field type is " + xvalue.getClass().getName()
						+ " and cannot be casted");
			}
		}

		return this.converter.toJavaType(specificXValue);
	}

	/**
	 * @param xo
	 *            the XObject from which to get the fields value
	 * @return the current XValue or null
	 */
	@SuppressWarnings("unchecked")
	public V getXValue(final XReadableObject xo) {
		final XReadableField xf = xo.getField(this.fieldId);
		if (xf == null) {
			return null;
		}
		return (V) xf.getValue();
	}

	/**
	 * Set the value of this field within XObject xo to the given value of java
	 * type <J>
	 *
	 * @param actorId
	 *            used to set the value
	 * @param xo
	 *            where to write
	 * @param javaValue
	 *            which Java value to write
	 */
	public void setValue(final XId actorId, final XWritableObject xo, final J javaValue) {
		assert xo != null;
		if (javaValue == null) {
			/* Should remove */
			if (xo.hasField(this.fieldId)) {
				xo.removeField(this.fieldId);
			}
		} else {
			/* Should add */
			XWritableField xf = xo.getField(this.fieldId);
			if (xf == null) {
				xf = xo.createField(this.fieldId);
			}
			final XValue xvalue = this.converter.toXydraValue(javaValue);
			xf.setValue(xvalue);
		}
	}

	public XId getFieldId() {
		return this.fieldId;
	}

}
