package com.calpano.common.shared.util;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;

import org.xydra.log.api.Logger;
import org.xydra.log.api.LoggerFactory;

public class SharedUrl {

	@SuppressWarnings("unused")
	private static final Logger log = LoggerFactory.getLogger(SharedUrl.class);

	/* including colon */
	private String protocol = null;
	private String hostName = null;
	/* without colon */
	private String port = null;
	private String path;
	private String queryString;
	private String hash;

	public boolean isRelative() {
		return this.hostName == null;
	}

	/**
	 * @param protocol
	 * @param hostName
	 * @param port
	 * @param path
	 * @param queryString
	 * @param hashString
	 * @throws IllegalArgumentException
	 *             for parse errors
	 */
	public SharedUrl(final String protocol, final String hostName, final String port, final String path,
			final String queryString, final String hashString) throws IllegalArgumentException {
		this.protocol = protocol;
		this.hostName = hostName;
		this.port = port;
		if (port != null && this.port.contains(":")) {
			throw new IllegalArgumentException("Port syntax? '" + port + "'");
		}
		this.path = path;
		this.queryString = queryString;
		this.hash = hashString;
	}

	/**
	 * @param relativeUrl
	 *            path, queryString, hash
	 * @return the parsed relative URL
	 * @throws IllegalArgumentException
	 */
	public static SharedUrl fromRelativeUrl(final String relativeUrl) throws IllegalArgumentException {
		if (!relativeUrl.startsWith("/") && !relativeUrl.startsWith("?")
				&& !relativeUrl.startsWith("#") && !relativeUrl.equals("")) {
			throw new IllegalArgumentException("'" + relativeUrl + "'");
		}
		String path;
		String queryString;
		String hashString;

		final int qmarkPos = relativeUrl.indexOf("?");
		if (qmarkPos >= 0) {
			path = relativeUrl.substring(0, qmarkPos);
			final String queryHash = relativeUrl.substring(qmarkPos + 1);
			final int hashPos = queryHash.indexOf("#");
			if (hashPos >= 0) {
				queryString = queryHash.substring(0, hashPos);
				hashString = queryHash.substring(hashPos + 1);
			} else {
				queryString = queryHash;
				hashString = null;
			}
		} else {
			queryString = null;
			final int hashPos = relativeUrl.indexOf("#");
			if (hashPos >= 0) {
				path = relativeUrl.substring(0, hashPos);
				hashString = relativeUrl.substring(hashPos + 1);
			} else {
				path = relativeUrl;
				hashString = null;
			}
		}

		final SharedUrl cu = new SharedUrl(null, null, null, path, queryString, hashString);
		return cu;
	}

	/**
	 * @param fullUrl
	 * @return the parsed absolute URL
	 * @throws IllegalArgumentException
	 *             if href is invalid url syntax
	 */
	public static SharedUrl fromAbsoluteUrl(final String fullUrl) throws IllegalArgumentException {
		if (fullUrl == null) {
			throw new IllegalArgumentException("Uri must be non-null");
		}
		if (fullUrl.length() == 0) {
			throw new IllegalArgumentException("Uri must be non-empty");
		}
		// TODO remove leading userinfos
		int colonPos = fullUrl.indexOf(":");
		if (colonPos == -1) {
			throw new IllegalArgumentException("Uri must contains a colon");
		}
		final String scheme = fullUrl.substring(0, colonPos + 1);
		String hostPortPathQueryHash = fullUrl.substring(scheme.length());
		if (hostPortPathQueryHash.startsWith("//")) {
			hostPortPathQueryHash = hostPortPathQueryHash.substring(2);
		}
		String hostPort;
		String pathQueryHash;
		int pathStart = hostPortPathQueryHash.indexOf("/");
		if (pathStart == -1) {
			pathStart = hostPortPathQueryHash.indexOf("?");
		}
		if (pathStart == -1) {
			pathStart = hostPortPathQueryHash.indexOf("#");
		}
		if (pathStart == -1) {
			pathStart = hostPortPathQueryHash.length();
			hostPort = hostPortPathQueryHash.substring(0, pathStart);
			pathQueryHash = "";
		} else {
			hostPort = hostPortPathQueryHash.substring(0, pathStart);
			pathQueryHash = hostPortPathQueryHash.substring(pathStart);
		}
		colonPos = hostPort.indexOf(":");
		String host;
		String port;
		if (colonPos > 0) {
			host = hostPort.substring(0, colonPos);
			port = hostPort.substring(host.length() + 1);
		} else {
			host = hostPort;
			port = "";
		}

		final SharedUrl relative = fromRelativeUrl(pathQueryHash);

		final SharedUrl c = new SharedUrl(scheme, host, port, relative.getPath(), relative.getQuery(),
				relative.getFragment());
		return c;
	}

	public String getFragment() {
		return this.hash;
	}

	public String getQuery() {
		return this.queryString;
	}

	public String getPath() {
		return this.path;
	}

	/**
	 * @param otherPath
	 * @return true if both parts locate the same 'page', i.e. trailing slashes
	 *         are ignored
	 */
	public boolean locatesPath(final String otherPath) {
		assert !this.path.contains("?");
		assert !this.path.contains("#");
		assert !otherPath.contains("?");
		assert !otherPath.contains("#");
		return removeTrailingSlash(this.path).equals(removeTrailingSlash(otherPath));
	}

	public static String removeTrailingSlash(final String path) {
		if (path.endsWith("/")) {
			return path.substring(0, path.length() - 1);
		} else {
			return path;
		}
	}

	public void setPath(final String targetPath) {
		this.path = removeTrailingSlash(targetPath);
	}

	public String getPort() {
		return this.port;
	}

	/**
	 * @return the resulting URL as a string
	 */
	@Override
	public String toString() {
		final StringBuilder sb = new StringBuilder();
		if (!isRelative()) {
			sb.append(this.protocol);
			// omit for mailto: urls
			if (this.protocol.startsWith("http")) {
				sb.append("//");
			}
			sb.append(this.hostName);
			if (this.port != null && this.port.length() > 0) {
				sb.append(":").append(this.port);
			}
		}
		sb.append(this.path);
		if (this.queryString != null && this.queryString.length() > 0) {
			sb.append("?").append(this.queryString);
		}
		if (this.hash != null && this.hash.length() > 0) {
			sb.append("#").append(this.hash);
		}
		final String result = sb.toString();
		assert !result.contains("//") || result.startsWith("http://")
				|| result.startsWith("https://") || result.startsWith("mailto:") : "url is '"
				+ result + "' and contains neither http(s):// nor mailto: debug=" + toDebug();
		return result;
	}

	public String toDebug() {
		return "scheme=" + this.protocol + " * hostname=" + this.hostName + " * port=" + this.port
				+ " * path=" + this.path + " * query=" + this.queryString + " * hash=" + this.hash;
	}

	/**
	 * @param name
	 * @param value
	 *            will NOT be urlencoded
	 */
	public void addQueryParam(final String name, final String value) {
		if (this.queryString == null || this.queryString.length() == 0) {
			this.queryString = name + "=" + value;
		} else {
			this.queryString += "&" + name + "=" + value;
		}
	}

	public void removeQueryParam(final String name) {
		if (this.queryString == null || this.queryString.length() == 0) {
			return;
		} else {
			final Map<String, String> queryParams = parseQueryString(getQueryString());
			queryParams.remove(name);
			this.queryString = toQueryString(queryParams);
		}
	}

	/**
	 * Performs not url de/encoding
	 *
	 * @param map
	 * @return a query string without a leading '?'
	 */
	public static String toQueryString(final Map<String, String> map) {
		final StringBuilder b = new StringBuilder();

		final Iterator<Entry<String, String>> it = map.entrySet().iterator();
		while (it.hasNext()) {
			final Entry<String, String> e = it.next();
			b.append(e.getKey()).append("=").append(e.getValue());
			if (it.hasNext()) {
				b.append("&");
			}
		}

		return b.toString();
	}

	/**
	 * Does not do any URL en/decoding
	 *
	 * @param queryString
	 *            @CanBeNull
	 * @return a map of parameter key --&gt; parameter value; the value can be
	 *         null. @NeverNull
	 */
	public static Map<String, String> parseQueryString(final String queryString) {
		final Map<String, String> map = new HashMap<String, String>();

		if (queryString != null && !queryString.equals("")) {
			final String[] parts = queryString.split("[&]");
			if (parts != null && parts.length > 0) {
				for (final String part : parts) {
					// take first '='
					final int equalSign = part.indexOf("=");
					if (equalSign > 0) {
						// kev-value
						final String key = part.substring(0, equalSign);
						final String value = part.substring(equalSign + 1, part.length());
						map.put(key, value);
					} else {
						// key-only
						map.put(part, null);
					}
				}
			}
		}

		return map;
	}

	/**
	 * @return query string without leading '?'
	 */
	public String getQueryString() {
		return this.queryString;
	}

	/**
	 * @param href
	 * @return ...
	 * @throws IllegalArgumentException
	 */
	public static SharedUrl fromAnyUrl(final String href) throws IllegalArgumentException {
		try {
			return fromAbsoluteUrl(href);
		} catch (final IllegalArgumentException e) {
			// try relative
			return fromRelativeUrl(href);
		}
	}

	public boolean hasPort(final String port) {
		return this.port != null && this.port.equals(port);
	}

	public void setQueryString(final String queryString) {
		this.queryString = queryString;
	}

	public void setHash(final String hash) {
		this.hash = hash;
	}

	public String getHostName() {
		return this.hostName;
	}

	public String getProtocol() {
		return this.protocol;
	}

}
