package com.calpano.common.shared.user.wrapper;

import java.util.Collections;

import org.xydra.base.Base;
import org.xydra.base.XId;
import org.xydra.base.rmof.XReadableObject;
import org.xydra.base.rmof.XWritableModel;
import org.xydra.base.rmof.XWritableObject;
import org.xydra.log.api.Logger;
import org.xydra.log.api.LoggerFactory;

import com.calpano.common.shared.user.IViralUser;
import com.calpano.common.shared.xydrautils.IBasedOnXWritableObject;
import com.calpano.common.shared.xydrautils.field.FieldProperty;
import com.calpano.common.shared.xydrautils.impl.BasedOnXWritableObject;

/**
 * A viral user is a wrapper on an {@link XWritableObject} to access.
 *
 * A second object is used to model a set of XIDs.
 *
 * See {@link IViralUser}.
 *
 * <ul>
 * <li>A set of direct successors (XID), can handle thousands of entries</li>
 * <li>Number of direct successors (managed as a different property, for speed)</li>
 * <li>previousHop ID</li>
 * <li>offerID</li>
 * <li>number of hops to root hop</li>
 * </ul>
 *
 * Space estimate: 4 fields (à 20 chars) + values: 3 x long (4 bytes), 1 x xid
 * (20 chars) = 80 chars + 24 bytes + 20 chars = 320 bytes + 24 bytes + 80 bytes
 * = 424 bytes
 *
 * @author xamde
 */
public class ViralUser extends BasedOnXWritableObject implements IViralUser,
		IBasedOnXWritableObject {

	private static final long serialVersionUID = 1L;

	/**
	 * A purely tracking related property. Slightly redundant with
	 * 'successorHopSetObject', but faster
	 */
	protected static FieldProperty<Long> _directSuccessorCount;

	/** A purely tracking related property. */
	protected static FieldProperty<Long> _numberOfHopsFromRoot;

	/** A purely tracking related property. */
	protected static FieldProperty<XId> _previousHop;

	/** A purely tracking related property. */
	protected static FieldProperty<Long> _totalSuccessorCount;

	@Override
	public long getDirectSuccessorCount() {
		lazyInitDirectSuccessorCount();
		return _directSuccessorCount.getValue(getXObject());
	}

	public XId getSuccessorHopSetObjectId() {
		return toSuccesorSetId(getId());
	}

	private static XId toSuccesorSetId(final XId hopId) {
		return Base.toId("_" + hopId + "-successors");
	}

	@Override
	public Iterable<XId> getDirectSuccessorIds(final XReadableObject successorHopSetObject) {
		if (successorHopSetObject == null) {
			return Collections.emptySet();
		} else {
			return successorHopSetObject;
		}
	}

	@Override
	public long getNumberOfHopsFromRoot() {
		lazyInitNumberOfHopsFromRoot();
		return _numberOfHopsFromRoot.getValue(getXObject());
	}

	@Override
	public XId getPreviousHopId() {
		lazyInitPreviousHop();
		return _previousHop.getValue(getXObject());
	}

	@Override
	public long getTotalSuccessorCount() {
		lazyInitTotalSuccessorCount();
		return _totalSuccessorCount.getValue(getXObject());
	}

	protected void lazyInitDirectSuccessorCount() {
		if (_directSuccessorCount == null) {
			_directSuccessorCount = new FieldProperty<Long>("directSuccessorCount", Long.class);
		}
	}

	protected void lazyInitNumberOfHopsFromRoot() {
		if (_numberOfHopsFromRoot == null) {
			_numberOfHopsFromRoot = new FieldProperty<Long>("numberOfHopsFromRoot", Long.class);
		}
	}

	protected void lazyInitPreviousHop() {
		if (_previousHop == null) {
			_previousHop = new FieldProperty<XId>("previousHop", XId.class);
		}
	}

	protected void lazyInitTotalSuccessorCount() {
		if (_totalSuccessorCount == null) {
			_totalSuccessorCount = new FieldProperty<Long>("totalSuccessorCount", Long.class);
		}
	}

	/**
	 * @return true if this user is the root user
	 */
	@Override
	public boolean isRootUser() {
		return getNumberOfHopsFromRoot() == 0 && getDirectSuccessorCount() > 0;
	}

	@SuppressWarnings("unused")
	private static final Logger log = LoggerFactory.getLogger(ViralUser.class);

	public ViralUser(final IBasedOnXWritableObject proto) {
		super(proto);
	}

	@Override
	public void addDirectSuccessorId(final XWritableModel model, final XId directSuccessor) {
		assert model.hasObject(getId());
		final XWritableObject succObject = model.createObject(getSuccessorHopSetObjectId());
		succObject.createField(directSuccessor);
	}

	@Override
	public void removeDirectSuccessorId(final XWritableModel model, final XId directSuccessor) {
		assert model.hasObject(getId());
		final XId succId = getSuccessorHopSetObjectId();
		final XWritableObject succObject = model.getObject(succId);
		if (succObject != null) {
			succObject.removeField(directSuccessor);
		}
	}

	@Override
	public void setDirectSuccesorCount(final long directSuccesorCount) {
		lazyInitDirectSuccessorCount();
		_directSuccessorCount.setValue(getActorId(), getXWritableObject(), directSuccesorCount);
	}

	@Override
	public void setNumberOfHopsFromRoot(final long numberOfHopsFromRoot) {
		lazyInitNumberOfHopsFromRoot();
		_numberOfHopsFromRoot.setValue(getActorId(), getXWritableObject(), numberOfHopsFromRoot);
	}

	@Override
	public void setPreviousHopId(final XId previousHopId) {
		lazyInitPreviousHop();
		_previousHop.setValue(getActorId(), getXWritableObject(), previousHopId);
	}

	@Override
	public void setTotalSuccessorCount(final long totalSuccessorCount) {
		lazyInitTotalSuccessorCount();
		_totalSuccessorCount.setValue(getActorId(), getXWritableObject(), totalSuccessorCount);
	}

	@Override
	public XId getObjectIdToLoad() {
		return getSuccessorHopSetObjectId();
	}
}
