package com.calpano.common.shared.mail;

import java.io.UnsupportedEncodingException;
import java.util.ArrayList;

import org.xydra.annotations.CanBeNull;
import org.xydra.annotations.NeverNull;
import org.xydra.annotations.ReadOperation;
import org.xydra.annotations.RunsInGWT;
import org.xydra.index.query.Pair;
import org.xydra.log.api.Logger;
import org.xydra.log.api.LoggerFactory;
import org.xydra.sharedutils.XyAssert;

import de.xam.texthtml.text.TextRenderer;

/**
 * Utils for working with email addresses, RFC 1522 (quoted-printable)
 *
 * @author xamde
 */
@RunsInGWT(true)
public class EmailUtils {

	private static final Logger log = LoggerFactory.getLogger(EmailUtils.class);

	/**
	 * @param potentialEmailAddress
	 *            never null
	 * @return true if the potentialEmailAddress looks like an email address
	 */
	@ReadOperation
	public static boolean isValidEmail(final String potentialEmailAddress) {
		return potentialEmailAddress != null && potentialEmailAddress.matches(TextRenderer.EMAIL);

		// IMPROVE 2012-04 check domain name
		// IMPROVE 2012-04 check MX records etc
	}

	public static String normalizeEmail(@NeverNull final String raw) {
		assert raw != null;
		String result = raw.toLowerCase();
		result = result.replace("@googlemail.com", "@gmail.com");
		return result;
	}

	/**
	 * @param email1
	 *            @CanBeNull
	 * @param email2
	 *            @CanBeNull
	 * @return if both email addresses are not null and address the same
	 *         recipient
	 */
	public static boolean isSameRecipient(@CanBeNull final String email1, @CanBeNull final String email2) {
		return email1 != null && email2 != null
				&& normalizeEmail(email1).equals(normalizeEmail(email2));
	}

	/**
	 * @param s
	 * @return input with all RFC1522 encoded parts replaced by their encoded
	 *         unicode equivalents
	 */
	public static String decodeRfc1522(final String s) {
		String left = s;
		final StringBuilder result = new StringBuilder();

		int start = left.indexOf("=?");
		while (start >= 0) {
			result.append(s.substring(0, start));
			left = left.substring(start + "=?".length());
			final String[] parts = left.split("[?]");
			if (parts.length < 3) {
				// failed
				return s;
			}
			final String charset = parts[0].toLowerCase();
			final String encoding = parts[1].toLowerCase();
			String encodedText = parts[2];

			if (encoding.equals("q")) {
				try {
					encodedText = rfc1522qDecode(encodedText, charset);
					result.append(encodedText);
				} catch (final UnsupportedEncodingException e) {
					e.printStackTrace();
				}
			}

			final int end = left.indexOf("?=");
			if (end < 0) {
				// failed
				return s;
			}
			left = left.substring(end + "?=".length());
			start = left.indexOf("=?");
		}
		result.append(left);
		return result.toString();
	}

	/**
	 * @param s
	 *            in us-ascii encoding, contains '=F3' where F3 can be any hex
	 *            pair.
	 * @param charset
	 * @return decoded string as described by RFC1522, encoding 'Q'
	 * @throws UnsupportedEncodingException
	 */
	public static String rfc1522qDecode(final String s, final String charset)
			throws UnsupportedEncodingException {
		String r = s.replace("_", " ");

		// extract bytes
		final ArrayList<Byte> bytes = new ArrayList<Byte>();
		for (int i = 0; i < r.length(); i++) {
			final char c = r.charAt(i);
			if (c == '=') {
				if (i + 2 >= r.length()) {
					log.warn("mis-used '='");
					return r;
				}
				final char h1 = r.charAt(i + 1);
				final char h2 = r.charAt(i + 2);
				i = i + 2;
				final int d1 = fromHex(h1);
				final int d2 = fromHex(h2);
				final int digit = d1 * 16 + d2;
				bytes.add((byte) digit);
			} else {
				bytes.add((byte) c);
			}
		}
		final byte[] byteArray = new byte[bytes.size()];
		for (int i = 0; i < byteArray.length; i++) {
			byteArray[i] = bytes.get(i);
		}

		r = new String(byteArray, charset);

		return r;
	}

	private static int fromHex(final char c) {
		switch (c) {
		case '0':
			return 0;
		case '1':
			return 1;
		case '2':
			return 2;
		case '3':
			return 3;
		case '4':
			return 4;
		case '5':
			return 5;
		case '6':
			return 6;
		case '7':
			return 7;
		case '8':
			return 8;
		case '9':
			return 9;
		case 'a':
		case 'A':
			return 10;
		case 'b':
		case 'B':
			return 11;
		case 'c':
		case 'C':
			return 12;
		case 'd':
		case 'D':
			return 13;
		case 'e':
		case 'E':
			return 14;
		case 'f':
		case 'F':
			return 15;
		default:
			throw new IllegalArgumentException("Is not hex: '" + c + "'");
		}
	}

	public static void main(final String[] args) {
		final String s = "=?utf-8?q?Max_V=c3=b6lkel?= <max.voelkel@calpano.com>";
		System.out.println(s);
		System.out.println(decodeRfc1522(s));
		System.out.println("--");
		System.out.println(TextRenderer.EMAIL);
		assert isValidEmail("abc@web.de");
		assert isValidEmail("abc+def@web.de");
		assert isValidEmail("schorschmueller+label@gmail.com");
		assert isValidEmail("x@awaynowhere.de");
	}

	/**
	 * @param address
	 *            in format '"Jon Doe" <johnl@doemail.com>' or just
	 *            'john@doemail.com' or as bad as '=?utf-8?q?max_v=c3=b6lkel?=
	 *            <max.voelkel@example.com>' '
	 * @return name, email. Name can be null.
	 * @throws IllegalArgumentException
	 *             if syntax was violated
	 */
	@NeverNull
	public static Pair<String, String> parseAddress(final String address) {
		String s = address.trim();
		s = EmailUtils.decodeRfc1522(s);
		String name = "";
		String email = null;
		if (s.startsWith("\"")) {
			final int i = s.indexOf("\"", 1);
			if (i < 0) {
				throw new IllegalArgumentException("Address contains only one '\"' char");
			}
			assert i >= 0;
			name = s.substring(1, i);
			s = s.substring(i).trim();
		}
		// parse email
		final int i1 = s.indexOf('<');
		if (i1 >= 0) {
			final int i2 = s.indexOf(">");
			if (i2 < 0) {
				throw new IllegalArgumentException("Address contains '<' but no '>'");
			}
			name = name + s.substring(0, i1);
			email = s.substring(i1 + 1, i2);
		} else {
			email = s;
		}
		XyAssert.xyAssert(EmailUtils.isValidEmail(email), "not valid '%1' ", email);
		return new Pair<String, String>(name.trim(), email);
	}

}
