package com.calpano.common.shared.data;

import org.xydra.base.XAddress;
import org.xydra.core.model.impl.memory.IMemoryModel;
import org.xydra.log.api.Logger;
import org.xydra.log.api.LoggerFactory;
import org.xydra.store.XydraStore;
import org.xydra.store.sync.NewSyncer;
import org.xydra.store.sync.XSynchronizationCallback;

/**
 * Synchronizes changes between current application model and remote server
 * model.
 *
 * <br>
 * <br>
 * TODO register ajax calls
 *
 * @author alpha
 *
 */
public class Syncer {

	public interface SyncerCallback {
		/**
		 * The syncer failed.
		 *
		 * @param modelAddress
		 *            the model address of the syncer
		 *
		 * @param exception
		 *            the exception
		 */
		void onFailedSync(XAddress modelAddress, Throwable exception);

		/**
		 * A successful model sync
		 *
		 * @param model
		 *            the synced model
		 * @param hasChanges
		 *            true when the sync caused changes to the model
		 */
		void onSuccessfulSync(IMemoryModel model, boolean hasChanges);

	}

	private static final Logger log = LoggerFactory.getLogger(Syncer.class);

	private final IMemoryModel model;

	private final NewSyncer synchronizer;

	public Syncer(final IMemoryModel model, final XydraStore store) {
		this.model = model;
		this.synchronizer = new NewSyncer(store, model, model.getState(), model.getRoot(),
				model.getSessionActor(), model.getSessionPasswordHash(),
				model.getSynchronizedRevision());
	}

	public void synchronize(final SyncerCallback cb) {
		assert this.model != null;

		this.synchronizer.startSync(new XSynchronizationCallback() {
			// FIXME this is not threadsafe
			long syncRevBeforeSync = Syncer.this.model.getSynchronizedRevision();

			private boolean hasChanged() {
				return Syncer.this.model.getSynchronizedRevision() != this.syncRevBeforeSync;
			}

			@Override
			public void onCommandErrror(final Throwable t) {
				log.info("SyncCallback command error", t);
				cb.onFailedSync(Syncer.this.model.getAddress(), t);
			}

			@Override
			public void onEventsError(final Throwable t) {
				log.info("SyncCallback events error", t);
				cb.onFailedSync(Syncer.this.model.getAddress(), t);
			}

			@Override
			public void onRequestError(final Throwable t) {
				log.warn("SyncCallback request error", t);
				cb.onFailedSync(Syncer.this.model.getAddress(), t);
			}

			@Override
			public void onSuccess() {
				log.info("SyncCallback success");
				cb.onSuccessfulSync(Syncer.this.model, hasChanged());
			}
		});
	}
}
