package com.calpano.common.shared.data;

import java.util.List;

import org.xydra.base.XAddress;
import org.xydra.base.XId;
import org.xydra.base.rmof.XReadableModel;
import org.xydra.persistence.GetWithAddressRequest;
import org.xydra.store.BatchedResult;
import org.xydra.store.Callback;
import org.xydra.store.XydraStore;

/**
 *
 * Loads snapshots of models from a given store
 *
 * <br>
 * <br>
 * TODO register ajax calls
 *
 * Similar to {@link Syncer}.
 */
public class Snapshotizer {

	public interface SnapshotizerCallback {
		/**
		 * The particular snapshot failed for a single model
		 *
		 * @param modelAddress
		 *            the address of the failed model snapshot
		 *
		 * @param exception
		 *            the exception
		 */
		void onFailedSnapshot(XAddress modelAddress, Throwable exception);

		/**
		 * The snapshotizer failed as a whole. Because of no connection or
		 * server failure/offline or other reason.
		 *
		 * @param exception
		 *            the exception
		 */
		void onFailedSnapshotizer(Throwable exception);

		/**
		 * A snapshot for a model
		 *
		 * @param snapshot
		 *            the snapshot
		 */
		void onSuccessfulSnapshot(XReadableModel snapshot);

	}

	/** Where to get snapshots from */
	private final XydraStore store;

	/**
	 * see {@link Snapshotizer}
	 *
	 * @param store
	 */
	public Snapshotizer(final XydraStore store) {
		this.store = store;
	}

	private void getModelSnapshots(final XId actor, final String passwordHash,
			final Callback<BatchedResult<XReadableModel>[]> callback, final XAddress... modelAddresses) {
		final GetWithAddressRequest[] modelAddressRequests = new GetWithAddressRequest[modelAddresses.length];
		for (int i = 0; i < modelAddresses.length; i++) {
			modelAddressRequests[i] = new GetWithAddressRequest(modelAddresses[i]);
		}
		this.store.getModelSnapshots(actor, passwordHash, modelAddressRequests, callback);
	}

	public void getModelSnapshots(final XId actorId, final String passwordHash,
			final SnapshotizerCallback callback, final List<XAddress> modelAddresses) {
		getModelSnapshots(actorId, passwordHash, callback,
				modelAddresses.toArray(new XAddress[modelAddresses.size()]));
	}

	protected void getModelSnapshots(final XId actorId, final String passwordHash,
			final SnapshotizerCallback callback, final XAddress... modelAddresses) {
		final Callback<BatchedResult<XReadableModel>[]> getSnapshotsCallback = new Callback<BatchedResult<XReadableModel>[]>() {

			@Override
			public void onFailure(final Throwable exception) {
				callback.onFailedSnapshotizer(exception);
			}

			@Override
			public void onSuccess(final BatchedResult<XReadableModel>[] results) {
				for (int i = 0; i < results.length; i++) {
					final BatchedResult<XReadableModel> result = results[i];

					if (result.getException() == null) {
						final XReadableModel snapshot = result.getResult();
						callback.onSuccessfulSnapshot(snapshot);
					} else {
						callback.onFailedSnapshot(modelAddresses[i], result.getException());
					}
				}
			}
		};

		getModelSnapshots(actorId, passwordHash, getSnapshotsCallback, modelAddresses);

	}

}
