package com.calpano.common.shared.data;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import org.xydra.annotations.CanBeNull;
import org.xydra.annotations.NeverNull;
import org.xydra.log.api.Logger;
import org.xydra.log.api.LoggerFactory;

import com.calpano.common.shared.data.DataEvent.ActionKind;
import com.calpano.common.shared.data.DataEvent.DataEventHandler;
import com.google.web.bindery.event.shared.Event.Type;
import com.google.web.bindery.event.shared.EventBus;
import com.google.web.bindery.event.shared.HandlerRegistration;

import de.xam.p13n.shared.time.TimeProvider;

/**
 * Listens at the {@link EventBus} and stores data of<em>all specified</em> (not
 * all) {@link DataEvent} {@link Type}s.
 *
 * Data can be retrieved via {@link #getDataForEventType(Type)}
 *
 * Notifies of the availability of the data by firing {@link DataEvent}s with
 * kind {@link ActionKind#AvailableSuccess}
 *
 * @author alpha
 */
public class DataSink {

	/**
	 * A reference to a data object. Makes lazy resolving easier by first
	 * passing around the DataHolder reference and later setting the data.
	 *
	 * @param <T>
	 *            type
	 */
	public class DataHolder<T> {

		private long updateTime;

		/**
		 * @return the UTC time at which this DataHolder got the data updated.
		 */
		public long getUpdateTime() {
			return this.updateTime;
		}

		private T data = null;

		/**
		 * @return @CanBeNull
		 */
		public T getData() {
			return this.data;
		}

		/**
		 * @return true if data is not null
		 */
		public boolean hasData() {
			return this.data != null;
		}

		/**
		 * @param data
		 * @CanBeNull
		 */
		public void setData(final T data) {
			this.data = data;
			this.updateTime = TimeProvider.getCurrentTimeInMillis();
		}
	}

	@SuppressWarnings("unused")
	private static final Logger log = LoggerFactory.getLogger(DataSink.class);

	/** on which to listen */
	private final EventBus eventBus;

	/** data storage */
	private final Map<Type<?>, DataHolder<?>> holderMap = new ConcurrentHashMap<Type<?>, DataHolder<?>>();

	/** for which events do we listen? */
	private final Map<Type<?>, HandlerRegistration> typeRegistrationMap = new HashMap<Type<?>, HandlerRegistration>();

	/**
	 * Make sure to also {@link #registerEventType(Type)} for which to listen.
	 *
	 * @param eventBus
	 *            on which to listen
	 */
	public DataSink(final EventBus eventBus) {
		this.eventBus = eventBus;
	}

	/**
	 * Return the data stored for the specified type of {@link DataEvent}
	 *
	 * @param type
	 *            the type of {@link DataEvent} to get the stored data for
	 * @return the data that is stored for the specified type of
	 *         {@link DataEvent} @CanBeNull
	 */
	public @CanBeNull <T> T getDataForEventType(final Type<DataEventHandler<T>> type) {
		final DataHolder<T> dh = getDataHolder(type);
		assert dh != null;
		return dh.getData();
	}

	@SuppressWarnings("unchecked")
	@NeverNull
	private <T> DataHolder<T> getDataHolder(final Type<DataEventHandler<T>> type) {
		DataHolder<T> dh = (DataHolder<T>) this.holderMap.get(type);
		if (dh == null) {
			dh = createDataHolderForType(type);
		}
		return dh;
	}

	@NeverNull
	private <T> DataHolder<T> createDataHolderForType(final Type<DataEventHandler<T>> type) {
		final DataHolder<T> dp = new DataHolder<T>();
		this.holderMap.put(type, dp);
		return dp;
	}

	/**
	 * Registers a {@link Type} of {@link DataEvent} with this data sink. This
	 * data sink then listens on the {@link EventBus} for these type of
	 * {@link DataEvent}s and stores the data contained in {@link DataEvent}
	 * when the {@link ActionKind} of the {@link DataEvent} indicates a
	 * successful result. When stored, the {@link DataSink} notifies of the
	 * availability of the data by sending a new {@link DataEvent} with the same
	 * type and the {@link ActionKind#AvailableSuccess}.
	 *
	 * @param type
	 *            a {@link Type} of {@link DataEvent} to listen for on the
	 *            {@link EventBus} and store data for
	 */
	public synchronized <T> void registerEventType(final Type<DataEventHandler<T>> type) {
		if (!this.holderMap.containsKey(type)) {
			createDataHolderForType(type);
		}

		if (!this.typeRegistrationMap.containsKey(type)) {
			final HandlerRegistration reg = this.eventBus.addHandler(type, new DataEventHandler<T>() {

				@Override
				public void onData(final DataEvent<T> event) {
					/*
					 * Notify of _successful_ data events and store the data
					 * contained in them.
					 */
					if (event.getKind().isSuccess()) {
						setData(type, event.getData());
						/*
						 * Avoid infinite loop: Only notify of availability for
						 * events of kind not already Kind.Available.
						 */
						if (!event.getKind().equals(ActionKind.AvailableSuccess)) {
							final DataEvent<T> copyOfEvent = event
									.createCopyWithKind(ActionKind.AvailableSuccess);
							DataSink.this.eventBus.fireEvent(copyOfEvent);
						}
					}
				}
			});
			this.typeRegistrationMap.put(type, reg);
		}
	}

	/**
	 * Set data.
	 *
	 * @param type
	 *            the event type to store data for
	 * @param data
	 *            the data to store
	 */
	private <T> void setData(final Type<DataEventHandler<T>> type, final T data) {
		final DataHolder<T> dh = getDataHolder(type);
		dh.setData(data);
	}

	/**
	 * Unregisters a {@link Type} of {@link DataEvent}. This data sink then
	 * removes the listener on the {@link EventBus} for these type of
	 * {@link DataEvent}s.
	 *
	 * @param type
	 *            a {@link Type} of {@link DataEvent} to no longer listens for
	 */
	public synchronized <T> void unregisterEventType(final Type<DataEventHandler<T>> type) {
		if (this.typeRegistrationMap.containsKey(type)) {
			final HandlerRegistration reg = this.typeRegistrationMap.remove(type);
			reg.removeHandler();
		}
	}

}
