package com.calpano.common.client.view.forms.validation.impl;

import java.util.SortedMap;
import java.util.TreeMap;

import org.xydra.annotations.CanBeNull;
import org.xydra.annotations.NeverNull;
import org.xydra.log.api.Logger;
import org.xydra.log.api.LoggerFactory;

import com.calpano.common.client.ClientApp;
import com.calpano.common.client.view.forms.Html5DomUtil;
import com.calpano.common.client.view.forms.IHtml5TextInput;
import com.calpano.common.client.view.forms.ViewConstants;
import com.calpano.common.client.view.forms.validation.HasInvaliationHandlers;
import com.calpano.common.client.view.forms.validation.HasValidationHandlers;
import com.calpano.common.client.view.forms.validation.InvalidationEvent;
import com.calpano.common.client.view.forms.validation.InvalidationHandler;
import com.calpano.common.client.view.forms.validation.ValidationEvent;
import com.calpano.common.client.view.forms.validation.ValidationHandler;
import com.calpano.common.shared.validation.IStringValidator;
import com.calpano.common.shared.validation.TextValidator;
import com.calpano.common.shared.validation.ValidationMessage;
import com.calpano.common.shared.validation.ValidationMessage.Level;
import com.google.gwt.event.logical.shared.ValueChangeEvent;
import com.google.gwt.event.logical.shared.ValueChangeHandler;
import com.google.web.bindery.event.shared.HandlerRegistration;

/**
 * Calculates validation message and fires events.
 */
public class CalculateValidationHelper implements HasValidationHandlers, HasInvaliationHandlers {

	private static final Logger log = LoggerFactory.getLogger(CalculateValidationHelper.class);

	@CanBeNull
	private ValidationMessage currentValidationMessage;

	@CanBeNull
	private ValidationMessage lastFiredValidationMessage = null;

	private boolean required = false;

	private final boolean simulateValidation;

	private final IHtml5TextInput source;

	private String type = "";

	private boolean validation = false;

	public void setValidation(final boolean active) {
		this.validation = active;
	}

	/**
	 * Validators, sorted by number, usually from generic/cheap to compute to
	 * specific/time consuming
	 */
	private final SortedMap<Integer, IStringValidator> validators = new TreeMap<Integer, IStringValidator>();

	/**
	 * @param source
	 * @param simulateValidation
	 *            if true, we let GWT do the work. If false, we use native
	 *            HTML5.
	 *
	 *            IMPROVE implement also to use native html5.
	 */
	public CalculateValidationHelper(final IHtml5TextInput source, final boolean simulateValidation) {
		this.source = source;
		this.simulateValidation = simulateValidation;
		this.source.getTextBoxBase().addValueChangeHandler(new ValueChangeHandler<String>() {

			@Override
			public void onValueChange(final ValueChangeEvent<String> event) {
				onUserTypedTextChanged(event.getValue());
			}
		});
	}

	/**
	 * @param order
	 *            the closer to 1 the earlier it gets used
	 * @param validator
	 */
	public void addValidator(final Integer order, @NeverNull final IStringValidator validator) {
		this.validators.put(order, validator);
	}

	/**
	 * If the validation state changes between valid or invalid, events are
	 * fired.
	 *
	 * @param userTypedText
	 */
	void computeAndFireEvents(final String userTypedText) {
		computeValidation();

		if (!this.currentValidationMessage.equals(this.lastFiredValidationMessage)) {
			log.trace("firing now " + this.currentValidationMessage);
			fireValidityChangeEvent(this.currentValidationMessage);
		} else {
			log.trace("fired already " + this.lastFiredValidationMessage);
		}
	}

	private void fireValidityChangeEvent(final ValidationMessage m) {
		if (m.level.isValid()) {
			ClientApp.getEventBus().fireEventFromSource(new ValidationEvent(this.source),
					this.source);
		} else {
			ClientApp.getEventBus().fireEventFromSource(new InvalidationEvent(this.source),
					this.source);
		}
		this.lastFiredValidationMessage = m;
	}

	public String getType() {
		return this.type;
	}

	public ValidationMessage getValidationMessage() {
		return this.currentValidationMessage;
	}

	public boolean isRequired() {
		return this.required;
	}

	public void onLoad() {

		addValidator(2, new TextValidator(isRequired(), getType(),
				ViewConstants.INVALID_EMAIL_REQUIRED_BUT_EMPTY, ViewConstants.INVALID_BAD_EMAIL));
		// initial string might be non-empty
		final String userTypedText = this.source.getText();
		computeAndFireEvents(userTypedText);
	}

	private void onUserTypedTextChanged(final String userTypedText) {
		if (!this.validation) {
			return;
		}

		computeAndFireEvents(userTypedText);
	}

	public void reset() {
		this.lastFiredValidationMessage = null;
	}

	public void setRequired(final boolean required) {
		this.required = required;
		if (!this.simulateValidation) {
			Html5DomUtil.setPropertyString(this.source.getTextBoxBase(), "required", "required");
		}
	}

	public void setRequired(final String attributeValue) {
		if (attributeValue.equals("required") || attributeValue.equals("")) {
			setRequired(true);
		}
	}

	public void setType(final String type) {
		this.type = type != null ? type : "";
		if (!this.simulateValidation) {
			Html5DomUtil.setPropertyString(this.source.getTextBoxBase(), "type", type);
		}
	}

	@Override
	public HandlerRegistration addInvalidationHandler(final InvalidationHandler handler) {
		return ClientApp.getEventBus().addHandlerToSource(InvalidationEvent.TYPE, this.source,
				handler);
	}

	@Override
	public HandlerRegistration addValidationHandler(final ValidationHandler handler) {
		return ClientApp.getEventBus().addHandlerToSource(ValidationEvent.TYPE, this.source,
				handler);
	}

	/**
	 * @return the current, just updated validation message
	 */
	public ValidationMessage computeValidation() {
		final String userTypedText = this.source.getText();

		// find most severe problem
		ValidationMessage worst = ValidationMessage.ALL_OK;
		for (final IStringValidator v : this.validators.values()) {
			final ValidationMessage m = v.computeValidation(userTypedText);
			log.trace("Validator " + v.getClass() + " says: " + m);
			if (m.level.moreCriticalThan(worst.level)) {
				worst = m;
			}
			if (worst.level == Level.ErrorWrong) {
				break;
			}
		}
		this.currentValidationMessage = worst;

		log.trace("validating content '" + userTypedText + "' as " + this.currentValidationMessage);

		return this.currentValidationMessage;
	}

	public void onUnload() {
		this.validators.clear();
	}

}
