/*
 * Based on SuggestBox, License:
 *
 * Copyright 2009 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.calpano.common.client.view.forms.suggestion.impl;

import org.xydra.annotations.LicenseApache;
import org.xydra.log.api.Logger;
import org.xydra.log.api.LoggerFactory;

import com.calpano.common.client.ClientApp;
import com.calpano.common.client.util.SuggestionDisplay;
import com.calpano.common.client.view.forms.impl.IKeyboardAndValueEventHandler;
import com.calpano.common.client.view.forms.suggestion.HasHideSuggestionHandlers;
import com.calpano.common.client.view.forms.suggestion.HasShowSuggestionHandlers;
import com.calpano.common.client.view.forms.suggestion.HideSuggestionEvent;
import com.calpano.common.client.view.forms.suggestion.HideSuggestionHandler;
import com.calpano.common.client.view.forms.suggestion.ISuggestionDisplay;
import com.calpano.common.client.view.forms.suggestion.ShowSuggestionEvent;
import com.calpano.common.client.view.forms.suggestion.ShowSuggestionHandler;
import com.google.gwt.event.dom.client.KeyCodes;
import com.google.gwt.event.logical.shared.HasSelectionHandlers;
import com.google.gwt.event.logical.shared.SelectionEvent;
import com.google.gwt.event.logical.shared.SelectionHandler;
import com.google.gwt.event.logical.shared.ValueChangeEvent;
import com.google.gwt.event.shared.GwtEvent;
import com.google.gwt.event.shared.HandlerManager;
import com.google.gwt.user.client.Event;
import com.google.gwt.user.client.Timer;
import com.google.gwt.user.client.ui.HasEnabled;
import com.google.gwt.user.client.ui.SuggestBox;
import com.google.gwt.user.client.ui.SuggestOracle;
import com.google.gwt.user.client.ui.SuggestOracle.Callback;
import com.google.gwt.user.client.ui.SuggestOracle.Request;
import com.google.gwt.user.client.ui.SuggestOracle.Response;
import com.google.gwt.user.client.ui.SuggestOracle.Suggestion;
import com.google.gwt.user.client.ui.TextBoxBase;
import com.google.web.bindery.event.shared.HandlerRegistration;

/**
 * Init via
 * {@link com.calpano.common.client.view.forms.impl.Html5TextBoxAndAreaPart#suggestValuesFrom}
 *
 * See also {@link SuggestBox}.
 */
@LicenseApache(project = "SuggestBox", copyright = "Copyright 2009 Google Inc.")
public class SuggestionManager implements HasSelectionHandlers<Suggestion>, HasEnabled,
		HasShowSuggestionHandlers, HasHideSuggestionHandlers, IKeyboardAndValueEventHandler {

	private static Logger log = LoggerFactory.getLogger(SuggestionManager.class);

	/**
	 * The callback used when a user selects a {@link Suggestion}.
	 */
	public static interface SuggestionCallback {
		void onSuggestionSelected(Suggestion suggestion);
	}

	private final TextBoxBase textboxbase;

	private int limit = 20;
	private boolean selectsFirstItem = true;
	private final SuggestOracle oracle;
	private String currentText;
	private final ISuggestionDisplay display;

	/**
	 * @param oracle
	 * @param textBoxBase required for positioning of suggestionDisplay and as
	 *            event source
	 * @param suggestionDisplay
	 */
	public SuggestionManager(final SuggestOracle oracle, final TextBoxBase textBoxBase,
			final SuggestionDisplay suggestionDisplay) {
		this.oracle = oracle;
		this.display = suggestionDisplay;
		this.textboxbase = textBoxBase;
	}

	private final Callback callback = new Callback() {
		@Override
		public void onSuggestionsReady(final Request request, final Response response) {
			// If disabled while request was in-flight, drop it
			if (!isEnabled()) {
				return;
			}
			if (response.getSuggestions().size() > 0) {
				ShowSuggestionEvent.fire(SuggestionManager.this.getTextBoxBase());
			}
			SuggestionManager.this.display.showSuggestions(SuggestionManager.this,
					response.getSuggestions(), isAutoSelectEnabled(),
					SuggestionManager.this.suggestionCallback, response.hasMoreSuggestions(),
					response.getMoreSuggestionsCount());
		}
	};

	private final SuggestionCallback suggestionCallback = new SuggestionCallback() {
		@Override
		public void onSuggestionSelected(final Suggestion suggestion) {
			setNewSelection(suggestion);
		}
	};

	private boolean enabled = true;

	private HandlerManager handlerManager;

	/**
	 * Gets the limit for the number of suggestions that should be displayed for
	 * this box. It is up to the current {@link SuggestOracle} to enforce this
	 * limit.
	 *
	 * @return the limit for the number of suggestions
	 */
	public int getLimit() {
		return this.limit;
	}

	/**
	 * Gets the suggest box's
	 * {@link com.google.gwt.user.client.ui.SuggestOracle}.
	 *
	 * @return the {@link SuggestOracle}
	 */
	public SuggestOracle getSuggestOracle() {
		return this.oracle;
	}

	/**
	 * Returns whether or not the first suggestion will be automatically
	 * selected. This behavior is on by default.
	 *
	 * @return true if the first suggestion will be automatically selected
	 */
	public boolean isAutoSelectEnabled() {
		return this.selectsFirstItem;
	}

	/**
	 * Turns on or off the behavior that automatically selects the first
	 * suggested item. This behavior is on by default.
	 *
	 * @param selectsFirstItem Whether or not to automatically select the first
	 *            suggestion
	 */
	public void setAutoSelectEnabled(final boolean selectsFirstItem) {
		this.selectsFirstItem = selectsFirstItem;
	}

	/**
	 * Sets the limit to the number of suggestions the oracle should provide. It
	 * is up to the oracle to enforce this limit.
	 *
	 * @param limit the limit to the number of suggestions provided
	 */
	public void setLimit(final int limit) {
		this.limit = limit;
	}

	public void showSuggestions(final String query) {
		log.trace("Show suggestions for '" + query + "'");
		assert this.oracle != null;
		if (query.length() == 0) {
			this.oracle.requestDefaultSuggestions(new Request(null, this.limit), this.callback);
		} else {
			this.oracle.requestSuggestions(new Request(query, this.limit), this.callback);
		}
	}

	private void fireSuggestionEvent(final Suggestion selectedSuggestion) {
		SelectionEvent.fire(this, selectedSuggestion);
	}

	private void refreshSuggestions(final String text) {
		if (text.equals(this.currentText)) {
			return;
		} else {
			this.currentText = text;
		}
		showSuggestions(text);
	}

	/**
	 * Set the new suggestion in the text box.
	 *
	 * @param curSuggestion the new suggestion
	 */
	private void setNewSelection(final Suggestion curSuggestion) {
		assert curSuggestion != null : "suggestion cannot be null";
		this.currentText = curSuggestion.getReplacementString();
		this.textboxbase.setText(this.currentText);
		hideSuggestions();
		fireSuggestionEvent(curSuggestion);
	}

	@Override
	public void fireEvent(final GwtEvent<?> event) {
		ensureHandlers().fireEvent(event);
	}

	@Override
	public boolean isEnabled() {
		return this.enabled;
	}

	@Override
	public void setEnabled(final boolean enabled) {
		this.enabled = enabled;
	}

	@Override
	public com.google.gwt.event.shared.HandlerRegistration addSelectionHandler(
			final SelectionHandler<Suggestion> handler) {
		return ensureHandlers().addHandler(SelectionEvent.getType(), handler);
	}

	/**
	 * Creates the {@link HandlerManager} used by this Widget. You can override
	 * this method to create a custom {@link HandlerManager}.
	 *
	 * @return the {@link HandlerManager} you want to use
	 */
	protected HandlerManager createHandlerManager() {
		return new HandlerManager(this);
	}

	/**
	 * Ensures the existence of the handler manager.
	 *
	 * @return the handler manager
	 * */
	HandlerManager ensureHandlers() {
		return this.handlerManager == null ? this.handlerManager = createHandlerManager()
				: this.handlerManager;
	}

	public TextBoxBase getTextBoxBase() {
		return this.textboxbase;
	}

	/**
	 * Suggestion was selected, not yet submitted.
	 *
	 * @param s
	 */
	public void fireSuggestionActivated(final Suggestion s) {
		hideSuggestions();
		this.textboxbase.setText(s.getReplacementString());
		// FIXME make sure an onValueChangeEvent is fired with the replacement
		// string
	}

	@Override
	public HandlerRegistration addShowSuggestionHandler(final ShowSuggestionHandler handler) {
		return ClientApp.getEventBus().addHandlerToSource(ShowSuggestionEvent.TYPE,
				getTextBoxBase(), handler);
	}

	@Override
	public HandlerRegistration addHideSuggestionHandler(final HideSuggestionHandler handler) {
		return ClientApp.getEventBus().addHandlerToSource(HideSuggestionEvent.TYPE,
				getTextBoxBase(), handler);
	}

	@Override
	public void onBrowserEvent(final Event event) {
		switch (event.getTypeInt()) {
		case Event.ONKEYDOWN: {
			switch (event.getKeyCode()) {
			case KeyCodes.KEY_DOWN:
				SuggestionManager.this.display.moveSelectionDown();
				break;
			case KeyCodes.KEY_UP:
				SuggestionManager.this.display.moveSelectionUp();
				break;
			case KeyCodes.KEY_ENTER:
			case KeyCodes.KEY_TAB:
				final Suggestion suggestion = SuggestionManager.this.display.getCurrentSelection();
				if (suggestion == null) {
					SuggestionManager.this.hideSuggestions();
				} else {
					setNewSelection(suggestion);
				}
				break;
			}
		}
			break;
		case Event.ONBLUR: {
			if (event.getTypeInt() == Event.ONBLUR) {
				new Timer() {

					@Override
					public void run() {
						SuggestionManager.this.hideSuggestions();
					}
				}.schedule(150);
			}
		}
			break;
		}
	}

	private void hideSuggestions() {
		HideSuggestionEvent.fire(SuggestionManager.this.textboxbase);
		SuggestionManager.this.display.hideSuggestions();
	}

	@Override
	public void onValueChange(final ValueChangeEvent<String> event) {
		// After every user key input (which changed the text string), refresh
		// the popup's suggestions.
		refreshSuggestions(event.getValue());
	}

	public void reset() {
		hideSuggestions();
	}

}
